/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.zip;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;

import net.sourceforge.jaulp.io.StreamUtils;

/**
 * The Class Unzipper.
 *
 * @author Asterios Raptis
 * @version 1.0
 */
public class Unzipper
{

	/** The zip file. */
	private ZipFile zipFile;

	/** The to dir. */
	private File toDir;

	/** The password. */
	private String password;

	/**
	 * Instantiates a new unzipper.
	 */
	public Unzipper()
	{
	}

	/**
	 * Instantiates a new unzipper.
	 * 
	 * @param zipFile
	 *            the zip file
	 * @param toDir
	 *            the to dir
	 */
	public Unzipper(final ZipFile zipFile, final File toDir)
	{
		this.zipFile = zipFile;
		this.toDir = toDir;
	}

	/**
	 * Extract zip entry.
	 * 
	 * @param zipFile
	 *            the zip file
	 * @param target
	 *            the target
	 * @param toDirectory
	 *            the to directory
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void extractZipEntry(final ZipFile zipFile, final ZipEntry target, final File toDirectory)
		throws IOException
	{
		final File fileToExtract = new File(toDirectory, target.getName());
		BufferedInputStream bis = null;
		BufferedOutputStream bos = null;
		InputStream is = null;
		try
		{
			is = zipFile.getInputStream(target);
			if (is != null)
			{
				bis = new BufferedInputStream(is);

				new File(fileToExtract.getParent()).mkdirs();

				bos = new BufferedOutputStream(new FileOutputStream(fileToExtract));
				for (int c; (c = bis.read()) != -1;)
				{
					bos.write((byte)c);
				}

				bos.close();
			}
		}
		catch (final IOException e)
		{
			throw e;
		}
		finally
		{
			StreamUtils.closeInputStream(is);
			StreamUtils.closeInputStream(bis);
			StreamUtils.closeOutputStream(bos);
		}
	}

	/**
	 * Returns the field <code>toDir</code>.
	 * 
	 * @return The field <code>toDir</code>.
	 */
	public File getToDir()
	{
		return this.toDir;
	}

	/**
	 * Returns the field <code>zipFile</code>.
	 * 
	 * @return The field <code>zipFile</code>.
	 */
	public ZipFile getZipFile()
	{
		return this.zipFile;
	}

	/**
	 * Sets the field <code>toDir</code>.
	 * 
	 * @param toDir
	 *            The <code>toDir</code> to set
	 */
	public void setToDir(final File toDir)
	{
		this.toDir = toDir;
	}

	/**
	 * Sets the field <code>zipFile</code>.
	 * 
	 * @param zipFile
	 *            The <code>zipFile</code> to set
	 */
	public void setZipFile(final ZipFile zipFile)
	{
		this.zipFile = zipFile;
	}

	/**
	 * Gets the password.
	 *
	 * @return the password
	 */
	public String getPassword()
	{
		return password;
	}

	/**
	 * Sets the password.
	 *
	 * @param password
	 *            the new password
	 */
	public void setPassword(String password)
	{
		this.password = password;
	}

	/**
	 * Unzip.
	 * 
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void unzip() throws IOException
	{
		this.unzip(this.zipFile, this.toDir);
	}

	/**
	 * Unzip.
	 * 
	 * @param zipFile
	 *            the zip file
	 * @param toDir
	 *            the to dir
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void unzip(final ZipFile zipFile, final File toDir) throws IOException
	{
		for (final Enumeration<? extends ZipEntry> e = zipFile.entries(); e.hasMoreElements();)
		{
			final ZipEntry entry = e.nextElement();
			this.extractZipEntry(zipFile, entry, toDir);
		}
		zipFile.close();
	}

	/**
	 * Unzip.
	 *
	 * @param zipFile
	 *            the zip file
	 * @param toDir
	 *            the to dir
	 * @param password
	 *            the password
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void unzip(final File zipFile, final File toDir, String password) throws IOException
	{
		unzip(zipFile, toDir, password, null);
	}

	/**
	 * Unzip.
	 *
	 * @param zipFile
	 *            the zip file
	 * @param toDir
	 *            the to dir
	 * @param password
	 *            the password
	 * @param charsetName
	 *            the charset name
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public void unzip(final File zipFile, final File toDir, String password, Charset charsetName)
		throws IOException
	{
		FileInputStream fis = new FileInputStream(zipFile);
		// wrap it in the decrypt stream
		ZipDecryptInputStream zdis = new ZipDecryptInputStream(fis, password);
		// wrap the decrypt stream by the ZIP input stream
		ZipInputStream zis = null;
		if (charsetName != null)
		{
			zis = new ZipInputStream(zdis, charsetName);
		}
		else
		{
			zis = new ZipInputStream(zdis);
		}
		// read all the zip entries and save them as files
		ZipEntry ze;
		while ((ze = zis.getNextEntry()) != null)
		{
			String pathToExtract = toDir.getAbsolutePath() + File.separator + ze.getName();
			FileOutputStream fos = new FileOutputStream(pathToExtract);
			int b;
			while ((b = zis.read()) != -1)
			{
				fos.write(b);
			}
			fos.close();
			zis.closeEntry();
		}
		zis.close();
	}

}
