/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.search;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * The Class PathFinder is a helper class for getting source folders from maven projects.
 */
public final class PathFinder
{
	/**
	 * Private constructor.
	 */
	private PathFinder()
	{
		super();
	}

	/**
	 * The Constant SOURCE_FOLDER_SRC_MAIN_RESOURCES keeps the relative path for the source folder
	 * 'src/main/resources' in maven projects.
	 * */
	public static final String SOURCE_FOLDER_SRC_MAIN_RESOURCES = "src/main/resources";

	/**
	 * The Constant SOURCE_FOLDER_SRC_MAIN_JAVA keeps the relative path for the source folder
	 * 'src/main/java' in maven projects.
	 * */
	public static final String SOURCE_FOLDER_SRC_MAIN_JAVA = "src/main/java";

	/**
	 * The Constant SOURCE_FOLDER_SRC_TEST_RESOURCES keeps the relative path for the source folder
	 * 'src/test/resources' in maven projects.
	 * */
	public static final String SOURCE_FOLDER_SRC_TEST_RESOURCES = "src/test/resources";

	/**
	 * The Constant SOURCE_FOLDER_SRC_TEST_JAVA keeps the relative path for the source folder
	 * 'src/test/java' in maven projects.
	 * */
	public static final String SOURCE_FOLDER_SRC_TEST_JAVA = "src/test/java";

	/**
	 * Gets the absolute path.
	 *
	 * @param file
	 *            the file
	 * @param removeLastChar
	 *            the remove last char
	 * @return the absolute path
	 */
	public static String getAbsolutePath(final File file, final boolean removeLastChar)
	{
		String absolutePath = file.getAbsolutePath();
		if (removeLastChar)
		{
			absolutePath = absolutePath.substring(0, absolutePath.length() - 1);
		}
		return absolutePath;
	}

	/**
	 * Gets the project directory.
	 *
	 * @param currentDir
	 *            the current dir
	 * @return the project directory
	 */
	public static File getProjectDirectory(final File currentDir)
	{
		final String projectPath = PathFinder.getAbsolutePath(currentDir, true);
		final File projectFile = new File(projectPath);
		return projectFile;
	}

	/**
	 * Gets the project directory.
	 *
	 * @return the project directory
	 */
	public static File getProjectDirectory()
	{
		return getProjectDirectory(new File("."));
	}

	/**
	 * Gets the src main resources dir.
	 *
	 * @param projectDirectory
	 *            the project directory
	 * @return the src main resources dir
	 */
	public static File getSrcMainResourcesDir(final File projectDirectory)
	{
		return new File(projectDirectory, PathFinder.SOURCE_FOLDER_SRC_MAIN_RESOURCES);
	}

	/**
	 * Gets the src main resources dir.
	 *
	 * @return the src main resources dir
	 */
	public static File getSrcMainResourcesDir()
	{
		return new File(getProjectDirectory(), PathFinder.SOURCE_FOLDER_SRC_MAIN_RESOURCES);
	}

	/**
	 * Gets the src main java dir.
	 *
	 * @param projectDirectory
	 *            the project directory
	 * @return the src main java dir
	 */
	public static File getSrcMainJavaDir(final File projectDirectory)
	{
		return new File(projectDirectory, PathFinder.SOURCE_FOLDER_SRC_MAIN_JAVA);
	}

	/**
	 * Gets the src main java dir.
	 *
	 * @return the src main java dir
	 */
	public static File getSrcMainJavaDir()
	{
		return new File(getProjectDirectory(), PathFinder.SOURCE_FOLDER_SRC_MAIN_JAVA);
	}


	/**
	 * Gets the src test resources dir.
	 *
	 * @param projectDirectory
	 *            the project directory
	 * @return the src test resources dir
	 */
	public static File getSrcTestResourcesDir(final File projectDirectory)
	{
		return new File(projectDirectory, PathFinder.SOURCE_FOLDER_SRC_TEST_RESOURCES);
	}

	/**
	 * Gets the src test resources dir.
	 *
	 * @return the src test resources dir
	 */
	public static File getSrcTestResourcesDir()
	{
		return new File(getProjectDirectory(), PathFinder.SOURCE_FOLDER_SRC_TEST_RESOURCES);
	}


	/**
	 * Gets the src test java dir.
	 *
	 * @param projectDirectory
	 *            the project directory
	 * @return the src test java dir
	 */
	public static File getSrcTestJavaDir(final File projectDirectory)
	{
		return new File(projectDirectory, PathFinder.SOURCE_FOLDER_SRC_TEST_JAVA);
	}

	/**
	 * Gets the src test java dir.
	 *
	 * @return the src test java dir
	 */
	public static File getSrcTestJavaDir()
	{
		return new File(getProjectDirectory(), PathFinder.SOURCE_FOLDER_SRC_TEST_JAVA);
	}

	/**
	 * Gets the file or directory from the given parent File object and the relative path given over
	 * the list as String objects.
	 *
	 * @param parent
	 *            The parent directory.
	 * @param folders
	 *            The list with the directories and optional a filename.
	 * @return the resulted file or directory from the given arguments.
	 */
	public static File getRelativePathTo(File parent, List<String> folders)
	{
		for (String string : folders)
		{
			File nextFolder = new File(parent, string);
			parent = nextFolder;
		}
		return parent;
	}

	/**
	 * Gets the file or directory from the given parent File object and the relative path given over
	 * the list as String objects.
	 *
	 * @param parent
	 *            The parent directory.
	 * @param folders
	 *            The list with the directories and optional a filename.
	 * @return the resulted file or directory from the given arguments.
	 */
	public static File getRelativePath(File parent, String... folders)
	{
		return getRelativePathTo(parent, Arrays.asList(folders));
	}

	/**
	 * Gets the file or directory from the given parent File object and the relative path given over
	 * the list as String objects.
	 *
	 * @param parent
	 *            The parent directory.
	 * @param separator
	 *            The separator for separate the String folders.
	 * @param folders
	 *            The relative path as a String object separated with the defined separator.
	 * @param filename
	 *            The filename.
	 * @return the resulted file or directory from the given arguments.
	 */
	public static File getRelativePathTo(File parent, String separator, String folders,
		String filename)
	{
		List<String> list = new ArrayList<>(Arrays.asList(folders.split(separator)));
		if (filename != null && !filename.isEmpty())
		{
			list.add(filename);
		}
		return getRelativePathTo(parent, list);
	}

}
