/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.properties;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import net.sourceforge.jaulp.file.copy.CopyFileUtils;
import net.sourceforge.jaulp.file.read.ReadFileUtils;
import net.sourceforge.jaulp.file.search.FileSearchUtils;
import net.sourceforge.jaulp.lang.PropertiesUtils;

public class PropertiesFileUtils
{

	/**
	 * Removes the comments from the given properties file.
	 * 
	 * @param propertiesFile
	 *            the properties file
	 * @return the list
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static List<String> removeComments(File propertiesFile) throws IOException
	{
		if (propertiesFile != null
			&& !propertiesFile.getName()
				.matches(FileSearchUtils.getSearchFilePattern("properties")))
		{
			throw new IllegalArgumentException("The given file is not an properties file.");
		}
		List<String> lines = ReadFileUtils.readLinesInList(propertiesFile);
		for (Iterator<String> itr = lines.iterator(); itr.hasNext();)
		{
			String line = itr.next();
			if (line.startsWith("#") || line.trim().length() == 0)
			{
				itr.remove();
			}
		}
		return lines;
	}

	/**
	 * Creates a backup file from the given properties file.
	 * 
	 * @param file
	 *            the file
	 * @return the backup file
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static File newBackupOf(File file) throws IOException
	{
		return CopyFileUtils.newBackupOf(file, Charset.forName("ISO-8859-1"),
			Charset.forName("UTF-8"));
	}

	/**
	 * Gets the redundant keys in properties files from the given directory. The search is recursive
	 * and finds all properties files. The result is a map with key the properties file and the
	 * found redundant keys as a map.
	 * 
	 * @param dirToSearch
	 *            the dir to search
	 * @return the redundant keys
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static Map<File, Map<String, List<String>>> getRedundantKeys(File dirToSearch)
		throws IOException
	{
		List<File> foundFiles = FileSearchUtils.findAllFiles(dirToSearch,
			FileSearchUtils.getSearchFilePattern("properties"));

		Map<String, List<String>> linesMap = new LinkedHashMap<>();
		Map<File, Map<String, List<String>>> fileMap = new LinkedHashMap<>();
		for (File file : foundFiles)
		{
			List<String> lines = PropertiesFileUtils.removeComments(file);
			Properties p = PropertiesUtils.loadProperties(file);

			for (Map.Entry<Object, Object> entry : p.entrySet())
			{
				String key = ((String)entry.getKey()).trim();
				for (String line : lines)
				{
					if (line.startsWith(key))
					{
						char nextChar = line.charAt(key.length());
						// char[] anc = {nextChar};
						// String nc = new String(anc);
						if (nextChar == '.')
						{
							continue;
						}
						else if (nextChar == '=' || nextChar == ':' || nextChar == ' ')
						{
							if (!linesMap.containsKey(key))
							{
								List<String> dl = new ArrayList<>();
								dl.add(line);
								linesMap.put(key, dl);
							}
							else
							{
								List<String> dl = linesMap.get(key);
								dl.add(line);
								linesMap.put(key, dl);
							}
						}
						else
						{
							throw new RuntimeException("nextChar is '" + nextChar + "'");
						}
					}
				}
			}

			Map<String, List<String>> duplicateKeys = new LinkedHashMap<>();
			for (Map.Entry<String, List<String>> entry : linesMap.entrySet())
			{
				if (1 < entry.getValue().size())
				{
					duplicateKeys.put(entry.getKey(), entry.getValue());
				}
			}
			if (0 < duplicateKeys.size())
			{
				fileMap.put(file, duplicateKeys);
			}
		}
		return fileMap;
	}

}
