/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.csv;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

/**
 * The Class CsvBean.
 */
public class CsvBean implements Serializable, Cloneable
{

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = 5428554488510583277L;


	/** The headers. */
	private String[] headers;

	/** The column types. */
	private String[] columnTypes;

	/** The lines. */
	private List<String[]> lines;

	/** The column types edit. */
	private String[] columnTypesEdit;


	/** The line order. */
	private Map<Integer, Integer> lineOrder;

	/**
	 * Gets the line order.
	 *
	 * @return the line order
	 */
	public Map<Integer, Integer> getLineOrder()
	{
		return lineOrder;
	}

	/**
	 * Sets the line order.
	 *
	 * @param lineOrder
	 *            the line order
	 */
	public void setLineOrder(Map<Integer, Integer> lineOrder)
	{
		this.lineOrder = lineOrder;
	}

	/**
	 * Instantiates a new csv bean.
	 */
	public CsvBean()
	{
		super();
	}

	/**
	 * Instantiates a new csv bean.
	 *
	 * @param headers
	 *            the headers
	 * @param columnTypes
	 *            the column types
	 * @param lines
	 *            the lines
	 */
	public CsvBean(final String[] headers, final String[] columnTypes, final List<String[]> lines)
	{
		this();
		this.headers = headers;
		this.columnTypes = columnTypes;
		this.lines = lines;
	}

	/**
	 * Instantiates a new csv bean.
	 *
	 * @param headers
	 *            the headers
	 * @param columnTypes
	 *            the column types
	 * @param columnTypesEdit
	 *            the column types edit
	 * @param lines
	 *            the lines
	 */
	public CsvBean(final String[] headers, final String[] columnTypes, String[] columnTypesEdit,
		final List<String[]> lines)
	{
		this();
		this.headers = headers;
		this.columnTypes = columnTypes;
		this.columnTypesEdit = columnTypesEdit;
		this.lines = lines;
	}

	/**
	 * Instantiates a new csv bean.
	 *
	 * @param headers
	 *            the headers
	 * @param columnTypes
	 *            the column types
	 * @param lines
	 *            the lines
	 * @param columnTypesEdit
	 *            the column types edit
	 * @param lineOrder
	 *            the line order
	 */
	public CsvBean(String[] headers, String[] columnTypes, List<String[]> lines,
		String[] columnTypesEdit, Map<Integer, Integer> lineOrder)
	{
		super();
		this.headers = headers;
		this.columnTypes = columnTypes;
		this.lines = lines;
		this.columnTypesEdit = columnTypesEdit;
		this.lineOrder = lineOrder;
	}

	/**
	 * Gets the column types edit.
	 *
	 * @return the column types edit
	 */
	public String[] getColumnTypesEdit()
	{
		return columnTypesEdit;
	}

	/**
	 * Sets the column types edit.
	 *
	 * @param columnTypesEdit
	 *            the new column types edit
	 */
	public void setColumnTypesEdit(String[] columnTypesEdit)
	{
		this.columnTypesEdit = columnTypesEdit;
	}


	/**
	 * Gets the headers.
	 *
	 * @return the headers
	 */
	public String[] getHeaders()
	{
		return headers;
	}

	/**
	 * Sets the headers.
	 *
	 * @param headers
	 *            the new headers
	 */
	public void setHeaders(final String[] headers)
	{
		this.headers = headers;
	}

	/**
	 * Gets the column types.
	 *
	 * @return the column types
	 */
	public String[] getColumnTypes()
	{
		return columnTypes;
	}

	/**
	 * Sets the column types.
	 *
	 * @param columnTypes
	 *            the new column types
	 */
	public void setColumnTypes(final String[] columnTypes)
	{
		this.columnTypes = columnTypes;
	}

	/**
	 * Gets the lines.
	 *
	 * @return the lines
	 */
	public List<String[]> getLines()
	{
		return lines;
	}

	/**
	 * Sets the lines.
	 *
	 * @param lines
	 *            the new lines
	 */
	public void setLines(final List<String[]> lines)
	{
		this.lines = lines;
	}

	/**
	 * (non-Javadoc).
	 *
	 * @return the object
	 * @see java.lang.Object#clone()
	 */
	@Override
	public Object clone()
	{
		CsvBean inst = new CsvBean();
		if (headers != null)
		{
			inst.headers = new String[headers.length];
			for (int i0 = 0; i0 < headers.length; i0++)
			{
				inst.headers[i0] = headers[i0] == null ? null : new String(headers[i0]);
			}
		}
		else
		{
			inst.headers = null;
		}
		if (columnTypes != null)
		{
			inst.columnTypes = new String[columnTypes.length];
			for (int i0 = 0; i0 < columnTypes.length; i0++)
			{
				inst.columnTypes[i0] = columnTypes[i0] == null ? null : new String(columnTypes[i0]);
			}
		}
		else
		{
			inst.columnTypes = null;
		}

		inst.lines = lines;
		return inst;
	}

	/**
	 * Returns <code>true</code> if this <code>CsvBean</code> is the same as the o argument.
	 *
	 * @param o
	 *            the o
	 * @return is the same as the o argument.
	 */
	@Override
	public boolean equals(final Object o)
	{
		if (this == o)
		{
			return true;
		}
		if (o == null)
		{
			return false;
		}
		if (o.getClass() != getClass())
		{
			return false;
		}
		CsvBean castedObj = (CsvBean)o;
		return java.util.Arrays.equals(headers, castedObj.headers)
			&& java.util.Arrays.equals(columnTypes, castedObj.columnTypes)
			&& (lines == null ? castedObj.lines == null : lines.equals(castedObj.lines));
	}

	/**
	 * Override hashCode.
	 *
	 * @return the Objects hashcode.
	 */
	@Override
	public int hashCode()
	{
		int hashCode = 1;
		hashCode = 31 * hashCode + (int)(+serialVersionUID ^ serialVersionUID >>> 32);
		for (int i0 = 0; headers != null && i0 < headers.length; i0++)
		{
			hashCode = 31 * hashCode + (headers == null ? 0 : headers[i0].hashCode());
		}
		for (int i0 = 0; columnTypes != null && i0 < columnTypes.length; i0++)
		{
			hashCode = 31 * hashCode + (columnTypes == null ? 0 : columnTypes[i0].hashCode());
		}
		hashCode = 31 * hashCode + (lines == null ? 0 : lines.hashCode());
		return hashCode;
	}

	/**
	 * (non-Javadoc).
	 *
	 * @return the string
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString()
	{
		StringBuilder buffer = new StringBuilder();
		buffer.append("[CsvBean:");
		buffer.append(" { ");
		for (int i0 = 0; headers != null && i0 < headers.length; i0++)
		{
			buffer.append(" headers[" + i0 + "]: ");
			buffer.append(headers[i0]);
		}
		buffer.append(" } ");
		buffer.append(" { ");
		for (int i0 = 0; columnTypes != null && i0 < columnTypes.length; i0++)
		{
			buffer.append(" columnTypes[" + i0 + "]: ");
			buffer.append(columnTypes[i0]);
		}
		buffer.append(" } ");
		buffer.append(" lines: ");
		buffer.append(lines);
		buffer.append("]");
		return buffer.toString();
	}

}
