/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.create;

import java.io.File;
import java.io.IOException;
import java.util.Collection;

import net.sourceforge.jaulp.file.exceptions.DirectoryAllreadyExistsException;

import org.apache.log4j.Logger;

/**
 * The Class CreateFileUtils helps you to create files or directories.
 *
 * @author Asterios Raptis
 * @version 1.0
 */
public final class CreateFileUtils
{

	/** The Constant logger. */
	private static final Logger logger = Logger.getLogger(CreateFileUtils.class.getName());

	/**
	 * Private constructor.
	 */
	private CreateFileUtils()
	{
		super();
	}

	/**
	 * Creates a new directory.
	 *
	 * @param dir
	 *            The directory to create.
	 *
	 * @return Returns true if the directory was created otherwise false.
	 *
	 * @throws DirectoryAllreadyExistsException
	 *             Thrown if the directory all ready exists.
	 */
	public static boolean newDirectory(final File dir) throws DirectoryAllreadyExistsException
	{
		boolean created = false;
		// If the directory does not exists
		if (!dir.exists())
		{ // then
			// create it...
			created = dir.mkdir();
		}
		else
		{ // otherwise
			// throw a DirectoryAllreadyExistsException.
			throw new DirectoryAllreadyExistsException("Directory allready exists.");
		}

		return created;
	}

	/**
	 * Creates a new directory quietly.
	 *
	 * @param dir
	 *            The directory to create.
	 *
	 * @return Returns true if the directory was created otherwise false.
	 */
	public static boolean newDirectoryQuietly(final File dir)
	{
		try
		{
			return newDirectory(dir);
		}
		catch (DirectoryAllreadyExistsException e)
		{
			logger.error("Directory '" + dir.getAbsolutePath() + "' allready exists.", e);
		}
		return false;
	}

	/**
	 * Creates the directories.
	 *
	 * @param directories
	 *            the directories
	 *
	 * @return true, if successful
	 */
	public static boolean newDirectoriesQuietly(Collection<File> directories)
	{
		boolean created = false;
		for (File dir : directories)
		{
			created = CreateFileUtils.newDirectoryQuietly(dir);
		}
		return created;
	}

	/**
	 * Creates the directories.
	 *
	 * @param directories
	 *            the directories
	 *
	 * @return true, if successful
	 *
	 * @throws DirectoryAllreadyExistsException
	 *             the directory allready exists exception
	 */
	public static boolean newDirectories(Collection<File> directories)
		throws DirectoryAllreadyExistsException
	{
		boolean created = false;
		for (File dir : directories)
		{
			created = CreateFileUtils.newDirectory(dir);
		}
		return created;

	}

	/**
	 * Creates an empty file if the File does not exists otherwise it lets the file as it is.
	 *
	 * @param file
	 *            the file.
	 *
	 * @return true, if the file is successful created otherwise false.
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static boolean newFile(final File file) throws IOException
	{
		boolean created = false;
		if (!file.exists())
		{
			newParentDirectories(file);
			created = file.createNewFile();
		}
		else
		{
			created = true;
		}
		return created;
	}

	/**
	 * Creates the parent directories from the given file.
	 *
	 * @param file
	 *            the file
	 */
	public static void newParentDirectories(final File file)
	{
		if (!file.exists())
		{
			File parent = file.getParentFile();
			if (parent != null && !parent.exists())
			{
				parent.mkdirs();
			}
		}
	}

	/**
	 * Creates an empty file quietly if the File does not exists otherwise it lets the file as it
	 * is.
	 *
	 * @param file
	 *            the file.
	 *
	 * @return true, if the file is successful created otherwise false.
	 */
	public static boolean newFileQuietly(final File file)
	{
		try
		{
			return newFile(file);
		}
		catch (IOException e)
		{
			logger.error("File '" + file.getAbsolutePath()
				+ "' could not created. For more information see the exception:", e);
		}
		return false;
	}

	/**
	 * Creates all files contained in the collection as empty files if the files does not exists
	 * otherwise it lets the files as they are.
	 *
	 * @param files
	 *            the Collection with the File objects.
	 *
	 * @return true, if successful
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	public static boolean newFiles(Collection<File> files) throws IOException
	{
		boolean created = false;
		for (File file : files)
		{
			created = CreateFileUtils.newFile(file);
		}
		return created;
	}

}
