/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file.compare;

import java.io.File;
import java.security.NoSuchAlgorithmException;

import net.sourceforge.jaulp.file.checksum.Algorithm;
import net.sourceforge.jaulp.file.checksum.ChecksumUtils;

/**
 * The Class SimpleCompareFileUtils.
 */
public final class SimpleCompareFileUtils
{

	/**
	 * Compare files by absolute path.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the absolute path are equal, otherwise false.
	 */
	public static boolean compareFilesByAbsolutePath(final File sourceFile, final File fileToCompare)
	{
		return CompareFileUtils.compareFiles(sourceFile, fileToCompare, false, true, true, true,
			true, true).getAbsolutePathEquality();
	}

	/**
	 * Compare files by content.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 *
	 * @return true if the content are equal, otherwise false.
	 */
	public static boolean compareFilesByContent(final File sourceFile, final File fileToCompare)
	{
		return CompareFileUtils.compareFiles(sourceFile, fileToCompare, true, true, true, true,
			true, false).getContentEquality();
	}

	/**
	 * Compare files by extension.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the extension are equal, otherwise false.
	 */
	public static boolean compareFilesByExtension(final File sourceFile, final File fileToCompare)
	{
		return CompareFileUtils.compareFiles(sourceFile, fileToCompare, true, false, true, true,
			true, true).getFileExtensionEquality();
	}

	/**
	 * Compare files by last modified.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the last modified are equal, otherwise false.
	 */
	public static boolean compareFilesByLastModified(final File sourceFile, final File fileToCompare)
	{
		return CompareFileUtils.compareFiles(sourceFile, fileToCompare, true, true, true, false,
			true, true).getLastModifiedEquality();
	}

	/**
	 * Compare files by length.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the length are equal, otherwise false.
	 */
	public static boolean compareFilesByLength(final File sourceFile, final File fileToCompare)
	{
		return CompareFileUtils.compareFiles(sourceFile, fileToCompare, true, true, false, true,
			true, true).getLengthEquality();
	}

	/**
	 * Compare files by name.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the name are equal, otherwise false.
	 */
	public static boolean compareFilesByName(final File sourceFile, final File fileToCompare)
	{
		return CompareFileUtils.compareFiles(sourceFile, fileToCompare, true, true, true, true,
			false, true).getNameEquality();
	}

	/**
	 * Compare files by checksum with the algorithm Adler32.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the checksum with the algorithm Adler32 are equal, otherwise false.
	 */
	public static boolean compareFilesByChecksumAdler32(final File sourceFile,
		final File fileToCompare)
	{
		long checksumSourceFile = ChecksumUtils.getCheckSumAdler32(sourceFile);
		long checksumFileToCompare = ChecksumUtils.getCheckSumAdler32(fileToCompare);
		return checksumSourceFile == checksumFileToCompare;
	}

	/**
	 * Compare files by checksum with the algorithm CRC32.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @return true if the checksum with the algorithm CRC32 are equal, otherwise false.
	 */
	public static boolean compareFilesByChecksumCRC32(final File sourceFile,
		final File fileToCompare)
	{
		long checksumSourceFile = ChecksumUtils.getCheckSumCRC32(sourceFile);
		long checksumFileToCompare = ChecksumUtils.getCheckSumCRC32(fileToCompare);
		return checksumSourceFile == checksumFileToCompare;
	}

	/**
	 * Compare files by checksum with the algorithm Adler32.
	 *
	 * @param sourceFile
	 *            the source file
	 * @param fileToCompare
	 *            the file to compare
	 * @param algorithm
	 *            the algorithm
	 * @return true if the checksum with the algorithm Adler32 are equal, otherwise false.
	 * @throws NoSuchAlgorithmException
	 *             Is thrown if the algorithm is not supported or does not exists.
	 */
	public static boolean compareFilesByChecksum(final File sourceFile, final File fileToCompare,
		Algorithm algorithm) throws NoSuchAlgorithmException
	{
		String checksumSourceFile = ChecksumUtils.getChecksum(sourceFile, algorithm);
		String checksumFileToCompare = ChecksumUtils.getChecksum(fileToCompare, algorithm);
		return checksumSourceFile.equals(checksumFileToCompare);
	}

	/**
	 * The Constructor.
	 */
	private SimpleCompareFileUtils()
	{
		super();
	}

}
