/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file;

import java.util.ArrayList;
import java.util.List;

/**
 * The Class GeneratorUtils.
 */
public final class GeneratorUtils
{
	/**
	 * Private constructor.
	 */
	private GeneratorUtils()
	{
		super();
	}

	/**
	 * Creates a String for a constant Stringarray to be inserted in java-file. For instance: List
	 * data = new ArrayList(); data.add("foo"); data.add("bar");
	 * newcreateStaticArrayVariable("test", data);
	 * Resultstring:"public static final String [] TEST = { "foo", "bar"};"
	 *
	 * @param arrayName
	 *            The name from the array.
	 * @param data
	 *            The data what to insert in the generated array.
	 * @return The result.
	 */
	public static String newcreateStaticArrayVariable(final String arrayName,
		final List<String> data)
	{
		final StringBuffer sb = new StringBuffer();
		sb.append("public static final String []");
		sb.append(arrayName.trim().toUpperCase());
		sb.append(" = {");
		final int size = data.size();
		for (int i = 0; i < size; i++)
		{
			final String element = data.get(i);
			if (i < size - 1)
			{
				sb.append("\"" + element + "\", ");
			}
			else
			{
				sb.append("\"" + element + "\"");
			}
		}
		sb.append("};");
		return sb.toString();
	}

	/**
	 * Creates a String list with the String constants from the given String list. For instance:
	 * List data = new ArrayList(); data.add("foo"); data.add("bar");
	 * newConstantsFromStringList(data, false); Result from
	 * list:["public static final String FOO = "foo";", "public static final String BAR = "bar";"]
	 *
	 * @param data
	 *            The data from what to create the contant strings.
	 * @param prefix
	 *            If the constant name needs a prefix.
	 * @param suffix
	 *            If the constant name needs a suffix.
	 * @param withQuotation
	 *            If the strings in the list have already quotation marks then true.
	 * @return A list with constants strings.
	 */
	public static List<String> newConstantsFromStringList(List<String> data, String prefix,
		String suffix, boolean withQuotation)
	{
		List<String> returnList = new ArrayList<>();

		final int size = data.size();
		for (int i = 0; i < size; i++)
		{
			final String element = data.get(i);
			StringBuilder sb = new StringBuilder();
			sb.append("public static final String ");
			if (withQuotation)
			{
				String striped = element.trim().toUpperCase();
				striped = striped.substring(1, striped.length() - 1);
				if (null != prefix)
				{
					sb.append(prefix.toUpperCase());
				}
				sb.append(striped);
				if (null != suffix)
				{
					sb.append(suffix.toUpperCase());
				}
				sb.append(" = ");
				sb.append(element + ";");
			}
			else
			{
				if (null != prefix)
				{
					sb.append(prefix.toUpperCase());
				}
				sb.append(element.trim().toUpperCase());
				if (null != suffix)
				{
					sb.append(suffix.toUpperCase());
				}
				sb.append(" = ");
				sb.append("\"" + element + "\";");
			}
			returnList.add(sb.toString().trim());
		}
		return returnList;

	}

}
