/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.file;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.net.URI;
import java.nio.channels.FileLock;

import net.sourceforge.jaulp.file.read.ReadFileUtils;
import net.sourceforge.jaulp.file.rename.RenameFileUtils;

import org.apache.log4j.Logger;

/**
 * Utility class for the use of File object. Most methods are set to deprecated and has gone to the
 * appropriate class.
 * 
 * @version 1.0
 * @author Asterios Raptis
 */
public final class FileUtils
{
	/**
	 * Private constructor.
	 */
	private FileUtils()
	{
		super();
	}

	/** The LOGGER. */
	private static final Logger LOGGER = Logger.getLogger(FileUtils.class.getName());

	/**
	 * Downloads Data from the given URI.
	 * 
	 * @param uri
	 *            The URI from where to download.
	 * @return Returns a byte array or null.
	 */
	public static byte[] download(final URI uri)
	{
		final File tmpFile = new File(uri);
		return ReadFileUtils.toByteArray(tmpFile);
	}

	/**
	 * Gets the absolut path without the filename.
	 * 
	 * @param file
	 *            the file.
	 * @return 's the absolut path without filename.
	 */
	public static String getAbsolutPathWithoutFilename(final File file)
	{
		return RenameFileUtils.getAbsolutPathWithoutFilename(file);
	}

	/**
	 * Gets the current absolut path without the dot and slash.
	 * 
	 * @return 's the current absolut path without the dot and slash.
	 */
	public static String getCurrentAbsolutPathWithoutDotAndSlash()
	{
		File currentAbsolutPath = new File(".");
		return currentAbsolutPath.getAbsolutePath().substring(0,
			currentAbsolutPath.getAbsolutePath().length() - 2);
	}

	/**
	 * Gets the filename with the absolute path prefix.
	 * 
	 * @param file
	 *            the file.
	 * @return the filename prefix.
	 */
	public static String getFilenamePrefix(final File file)
	{
		return FilenameUtils.getFilenamePrefix(file);
	}

	/**
	 * Gets the filename without the extension or null if the given file object is a directory.
	 * 
	 * @param file
	 *            the file.
	 * @return the filename without the extension or null if the given file object is a directory.
	 */
	public static String getFilenameWithoutExtension(final File file)
	{
		return FilenameUtils.getFilenameWithoutExtension(file);
	}

	/**
	 * Gets the filename suffix or null if no suffix exists or the given file object is a directory.
	 * 
	 * @param file
	 *            the file.
	 * @return 's the filename suffix or null if no suffix exists or the given file object is a
	 *         directory.
	 */
	public static String getFilenameSuffix(final File file)
	{
		return FilenameUtils.getFilenameSuffix(file);
	}

	/**
	 * Not yet implemented. Checks if the given file is open.
	 * 
	 * @param file
	 *            The file to check.
	 * @return Return true if the file is open otherwise false.
	 */
	public static boolean isOpen(final File file)
	{
		// TODO To implement with C or C++;
		boolean open = false;
		RandomAccessFile fileAccess = null;
		FileLock lock = null;
		try
		{
			fileAccess = new RandomAccessFile(file.getAbsolutePath(), "rw");
			lock = fileAccess.getChannel().tryLock();
			if (lock == null)
			{
				open = true;
			}
			else
			{
				lock.release();
			}
		}
		catch (FileNotFoundException fnfe)
		{
			open = true;
		}
		catch (IOException ioe)
		{
			open = true;
		}
		finally
		{
			if (fileAccess != null)
			{
				try
				{
					fileAccess.close();
					fileAccess = null;
				}
				catch (IOException ioex)
				{
					LOGGER.error(ioex);
				}
			}
		}
		return open;
	}

	/**
	 * Gets the temporary directory from the system as File object.
	 *
	 * @return the temporary directory from the system.
	 */
	public static File getTempDir()
	{
		return new File(System.getProperty("java.io.tmpdir"));
	}

}
