/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.export.excel.poi;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;

import net.sourceforge.jaulp.check.Check;

import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.WorkbookUtil;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

/**
 * Factory class for create poi objects.
 */
public class ExcelPoiFactory
{

	/**
	 * Creates a new CellStyle with the given date format.
	 *
	 * @param workbook
	 *            the workbook
	 * @param dateFormat
	 *            the date format
	 * @return the cell style
	 */
	public static CellStyle newDateCellStyle(Workbook workbook, String dateFormat)
	{
		CellStyle dateCellStyle = workbook.createCellStyle();
		dateCellStyle.setDataFormat(workbook.getCreationHelper().createDataFormat()
			.getFormat(dateFormat));
		return dateCellStyle;
	}

	/**
	 * Creates a new Sheet with the given name.
	 * <p>
	 * Note that sheet name is Excel must not exceed 31 characters and must not contain any of the
	 * any of the following characters:
	 * <ul>
	 * <li>0x0000</li>
	 * <li>0x0003</li>
	 * <li>colon (:)</li>
	 * <li>backslash (\)</li>
	 * <li>asterisk (*)</li>
	 * <li>question mark (?)</li>
	 * <li>forward slash (/)</li>
	 * <li>opening square bracket ([)</li>
	 * <li>closing square bracket (])</li>
	 * </ul>
	 * 
	 * @param workbook
	 *            the workbook
	 * @param name
	 *            the name
	 * @return the Sheet
	 */
	public static Sheet newSheet(Workbook workbook, String name)
	{
		return workbook.createSheet(WorkbookUtil.createSafeSheetName(name));
	}

	/**
	 * Creates a new CellStyle from the given parameters.
	 *
	 * @param workbook
	 *            the workbook
	 * @param fontName
	 *            the font name
	 * @param boldweight
	 *            the boldweight
	 * @param height
	 *            the height
	 * @return the cell style
	 */
	public static CellStyle newCellStyle(Workbook workbook, String fontName, short boldweight,
		short height)
	{
		CellStyle boldFontCellStyle = workbook.createCellStyle();
		boldFontCellStyle.setFont(newFont(workbook, fontName, boldweight, height));
		return boldFontCellStyle;
	}

	/**
	 * Creates a new font from the given parameters.
	 *
	 * @param workbook
	 *            the workbook
	 * @param fontName
	 *            the font name
	 * @param boldweight
	 *            the boldweight
	 * @param height
	 *            the height
	 * @return the font
	 */
	public static Font newFont(Workbook workbook, String fontName, short boldweight, short height)
	{
		Font font = workbook.createFont();
		font.setFontName(fontName);
		font.setBoldweight(boldweight);
		font.setFontHeightInPoints(height);
		return font;
	}

	/**
	 * Creates a new HSSFWorkbook from the given file name.
	 *
	 * @param pathname
	 *            the file name
	 * @return the Workbook
	 * @throws IOException
	 */
	public static Workbook newHSSFWorkbook(String pathname) throws IOException
	{
		Check.get().notNull(pathname, "pathname");
		return newHSSFWorkbook(new File(pathname));
	}

	/**
	 * Creates a new HSSFWorkbook from the given file.
	 *
	 * @param file
	 *            the file name
	 * @return the Workbook
	 * @throws IOException
	 */
	public static Workbook newHSSFWorkbook(File file) throws IOException
	{
		return writeWorkbook(new HSSFWorkbook(), file);
	}

	/**
	 * Creates a new HSSFWorkbook from the given file.
	 *
	 * @param file
	 *            the file name
	 * @return the Workbook
	 * @throws IOException
	 */
	public static Workbook newXSSFWorkbook(File file) throws IOException
	{
		return writeWorkbook(new XSSFWorkbook(), file);
	}

	/**
	 * Writes the given Workbook to the given file.
	 *
	 * @param workbook
	 *            the workbook
	 * @param file
	 *            the file
	 * @return the Workbook
	 * @throws IOException
	 */
	public static Workbook writeWorkbook(Workbook workbook, File file) throws IOException
	{
		FileOutputStream fileOutputStream = new FileOutputStream(file);
		workbook.write(fileOutputStream);
		fileOutputStream.close();
		return workbook;
	}

}
