/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.email.utils;

import java.io.UnsupportedEncodingException;

import javax.mail.Address;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;

import net.sourceforge.jaulp.email.messages.EmailConstants;
import net.sourceforge.jaulp.email.messages.EmailMessage;
import net.sourceforge.jaulp.string.StringUtils;

import org.apache.log4j.Logger;

/**
 * The Class EmailUtils provides methods for create email addresses and validate email addresses.
 */
public class EmailUtils
{

	/** The Constant logger. */
	private static final Logger logger = Logger.getLogger(EmailUtils.class.getName());

	/**
	 * Validate the given email address.
	 *
	 * @param emailAddress
	 *            the email address
	 *
	 * @return true, if successful
	 */
	public static boolean validateEmailAdress(String emailAddress)
	{
		boolean isValid = true;
		try
		{
			InternetAddress internetAddress = new InternetAddress(emailAddress);
			internetAddress.validate();
		}
		catch (AddressException e)
		{
			isValid = false;
		}
		return isValid;
	}

	/**
	 * Gets the encoding from the header.
	 *
	 * @param type
	 *            The string where the encoding is in.
	 *
	 * @return 's the encoding or null if its nothing found.
	 *
	 * @throws MessagingException
	 *             if an error occurs.
	 */
	public static String getCharsetFromContentType(String type) throws MessagingException
	{
		if (!StringUtils.isNullOrEmpty(type))
		{
			int start = type.indexOf(EmailConstants.CHARSET_PREFIX);
			if (start > 0)
			{
				start += EmailConstants.CHARSET_PREFIX.length();
				int offset = type.indexOf(" ", start);
				if (offset > 0)
				{
					return type.substring(start, offset);
				}
				else
				{
					return type.substring(start);
				}
			}
		}
		return null;
	}

	/**
	 * Creates from the given the address and personal name an Adress-object.
	 *
	 * @param emailAddress
	 *            The address in RFC822 format.
	 * @param personal
	 *            The personal name.
	 *
	 * @return The created Adress-object from the given address and personal name.
	 *
	 * @throws UnsupportedEncodingException
	 *             the unsupported encoding exception
	 * @throws AddressException
	 *             the address exception
	 */
	public static Address newAddress(String emailAddress, String personal) throws AddressException,
		UnsupportedEncodingException
	{
		return newAddress(emailAddress, personal, null);
	}

	/**
	 * Gets the address array from the given address objects.
	 *
	 * @param address
	 *            the address
	 * @return the address array
	 */
	public static Address[] getAddressArray(Address... address)
	{
		return address;
	}

	/**
	 * Creates an Address from the given the address and personal name.
	 *
	 * @param address
	 *            The address in RFC822 format.
	 * @param personal
	 *            The personal name.
	 * @param charset
	 *            MIME charset to be used to encode the name as per RFC 2047.
	 *
	 * @return The created InternetAddress-object from the given address and personal name.
	 *
	 * @throws AddressException
	 *             if an error occurs.
	 * @throws UnsupportedEncodingException
	 *             if an error occurs.
	 */
	public static Address newAddress(String address, String personal, String charset)
		throws AddressException, UnsupportedEncodingException
	{
		if (StringUtils.isNullOrEmpty(personal))
		{
			personal = address;
		}
		InternetAddress internetAdress = new InternetAddress(address);
		if (StringUtils.isNullOrEmpty(charset))
		{
			internetAdress.setPersonal(personal);
		}
		else
		{
			internetAdress.setPersonal(personal, charset);
		}
		return internetAdress;
	}

	/**
	 * Creates an Address from the given the email address as String object.
	 *
	 * @param address
	 *            The address in RFC822 format.
	 *
	 * @return The created InternetAddress-object from the given address.
	 *
	 * @throws AddressException
	 *             if an error occurs.
	 * @throws UnsupportedEncodingException
	 *             if an error occurs.
	 */
	public static Address newAddress(String address) throws AddressException,
		UnsupportedEncodingException
	{
		return newAddress(address, null, null);
	}

	/**
	 * Sets the from to the email message.
	 *
	 * @param senderEmail
	 *            the sender email
	 * @param senderPersonal
	 *            the sender personal
	 * @param senderCharset
	 *            the sender charset
	 * @param emailMessage
	 *            the email message
	 * @return the email message
	 */
	public static EmailMessage setFromToEmailMessage(final String senderEmail,
		final String senderPersonal, final String senderCharset, final EmailMessage emailMessage)
	{
		// Try to create the sender Address
		Address senderAddress = null;
		try
		{
			senderAddress = EmailUtils.newAddress(senderEmail, senderPersonal, senderCharset);

		}
		catch (final AddressException e2)
		{
			logger.error("AddressException when tryin to create the sender Address:" + senderEmail,
				e2);
			e2.printStackTrace();
		}
		catch (final UnsupportedEncodingException e2)
		{
			logger.error("UnsupportedEncodingException when tryin to create the sender Address:"
				+ senderEmail, e2);
			e2.printStackTrace();
		}
		// Set sender
		if (null != senderAddress)
		{
			try
			{
				emailMessage.setFrom(senderAddress);
			}
			catch (final MessagingException e)
			{
				try
				{
					emailMessage.setFrom(senderEmail);
				}
				catch (final AddressException e1)
				{
					logger.error("AddressException when tryin to set the From Address:"
						+ senderEmail, e1);
					e1.printStackTrace();
				}
				catch (final UnsupportedEncodingException e1)
				{
					logger.error("UnsupportedEncodingException when tryin to set the From Address:"
						+ senderEmail, e1);
					e1.printStackTrace();
				}
				catch (final MessagingException e1)
				{
					logger.error("MessagingException when tryin to set the From Address:"
						+ senderEmail, e1);
					e1.printStackTrace();
				}
				e.printStackTrace();
			}
		}
		else
		{
			try
			{
				emailMessage.setFrom(senderEmail);
			}
			catch (final AddressException e1)
			{
				logger.error("AddressException when tryin to set the From Address:" + senderEmail,
					e1);
				e1.printStackTrace();
			}
			catch (final UnsupportedEncodingException e1)
			{
				logger.error("UnsupportedEncodingException when tryin to set the From Address:"
					+ senderEmail, e1);
				e1.printStackTrace();
			}
			catch (final MessagingException e1)
			{
				logger.error(
					"MessagingException when tryin to set the From Address:" + senderEmail, e1);
				e1.printStackTrace();
			}
		}
		return emailMessage;
	}


	/**
	 * Adds a 'to' recipient to the email message.
	 *
	 * @param recipientEmail
	 *            the recipient email
	 * @param recipientPersonal
	 *            the recipient personal
	 * @param recipientCharset
	 *            the recipient charset
	 * @param emailMessage
	 *            the email message
	 * @return the email message
	 */
	public static EmailMessage addToRecipientToEmailMessage(final String recipientEmail,
		final String recipientPersonal, final String recipientCharset,
		final EmailMessage emailMessage)
	{

		// Try to create the recipient Address
		Address recipientAddress = null;
		try
		{
			recipientAddress = EmailUtils.newAddress(recipientEmail, recipientPersonal,
				recipientCharset);

		}
		catch (final AddressException e2)
		{
			logger.error("AddressException when tryin to create the recipient Address:"
				+ recipientEmail, e2);
			e2.printStackTrace();
		}
		catch (final UnsupportedEncodingException e2)
		{
			logger.error("UnsupportedEncodingException when tryin to create the recipient Address:"
				+ recipientEmail, e2);
			e2.printStackTrace();
		}

		// Set recipient
		if (null != recipientAddress)
		{
			try
			{
				emailMessage.addTo(recipientAddress);
			}
			catch (final MessagingException e)
			{
				try
				{
					emailMessage.setRecipients(Message.RecipientType.TO, recipientEmail);
				}
				catch (final AddressException e1)
				{
					logger.error("AddressException when tryin to set the To Address:"
						+ recipientEmail, e1);
					e1.printStackTrace();
				}
				catch (final MessagingException e1)
				{
					logger.error("MessagingException when tryin to set the To Address:"
						+ recipientEmail, e1);
					e1.printStackTrace();
				}
				e.printStackTrace();
			}
		}
		else
		{
			try
			{
				emailMessage.setRecipients(Message.RecipientType.TO, recipientEmail);
			}
			catch (final AddressException e1)
			{
				logger.error("AddressException when tryin to set the To Address:" + recipientEmail,
					e1);
				e1.printStackTrace();
			}
			catch (final MessagingException e1)
			{
				logger.error("MessagingException when tryin to set the To Address:"
					+ recipientEmail, e1);
				e1.printStackTrace();
			}
		}
		return emailMessage;
	}
}
