/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.email.send;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Properties;

import javax.mail.Address;
import javax.mail.Authenticator;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.AddressException;

import lombok.Getter;
import net.sourceforge.jaulp.email.messages.EmailConstants;
import net.sourceforge.jaulp.email.messages.EmailMessage;
import net.sourceforge.jaulp.email.messages.EmailMessageWithAttachments;
import net.sourceforge.jaulp.email.utils.EmailUtils;
import net.sourceforge.jaulp.lang.PackageUtils;
import net.sourceforge.jaulp.lang.PropertiesUtils;

/**
 * Helper-class for sending emails.
 *
 * @author Asterios Raptis
 */
public class SendEmail implements SendMail
{


	/** The debug. */
	@Getter
	private boolean debug;

	/** The properties file. */
	private Properties properties;

	/** The Session-object. */
	private Session session;

	private Authenticator authenticator;

	public Authenticator getAuthenticator()
	{
		return authenticator;
	}

	public void setAuthenticator(Authenticator authenticator)
	{
		this.authenticator = authenticator;
	}

	/**
	 * The Constructor.
	 */
	public SendEmail()
	{
		loadProperties();
	}

	/**
	 * The Constructor.
	 *
	 * @param properties
	 *            the properties
	 */
	public SendEmail(final Properties properties)
	{
		this.properties = properties;
	}

	/**
	 * The Constructor.
	 *
	 * @param properties
	 *            the properties
	 * @param authenticator
	 *            the authenticator
	 */
	public SendEmail(final Properties properties, Authenticator authenticator)
	{
		this.properties = properties;
		this.authenticator = authenticator;
	}

	/**
	 * Gets the Session-object.
	 *
	 * @return 's the Session-object.
	 */
	public synchronized Session getSession()
	{
		if (session == null)
		{
			if (authenticator != null)
			{
				session = Session.getInstance(properties, authenticator);
			}
			else
			{
				session = Session.getInstance(properties);
			}
		}
		return session;
	}

	/**
	 * Load properties.
	 */
	private void loadProperties()
	{
		final String packagePath = PackageUtils.getPackagePathWithSlash(this);
		final String propertiespath = packagePath + EmailConstants.PROPERTIES_FILENAME;
		try
		{
			properties = PropertiesUtils.loadProperties(this.getClass(), propertiespath);
		}
		catch (IOException e)
		{
			e.printStackTrace();
		}
		if (properties == null)
		{
			try
			{
				properties = PropertiesUtils.getLocalPropertiesFromClass(this.getClass(),
					this.getClass(), null);
			}
			catch (Exception e)
			{
				e.printStackTrace();
			}
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String sendEmailMessageWithAttachments(final EmailMessageWithAttachments emailAttachments)
		throws MessagingException
	{
		return sendEmailMessage(emailAttachments.getEmailMessage());
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<String> sendEmailMessagesWithAtachments(
		final Collection<EmailMessageWithAttachments> emailAttachments) throws MessagingException
	{
		List<String> messageIds = new ArrayList<>();
		for (EmailMessageWithAttachments emailAttachment : emailAttachments)
		{
			String messageId = sendEmailMessageWithAttachments(emailAttachment);
			messageIds.add(messageId);
		}
		return messageIds;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String sendEmailMessage(final EmailMessage emailMessage) throws MessagingException
	{
		Transport.send(emailMessage);
		return emailMessage.getMessageID();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Collection<String> sendEmailMessages(final Collection<EmailMessage> emailMessages)
		throws MessagingException
	{
		List<String> messageIds = new ArrayList<>();
		for (EmailMessage emailMessage : emailMessages)
		{
			String messageId = sendEmailMessage(emailMessage);
			messageIds.add(messageId);
		}
		return messageIds;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String sendEmail(final String to, final String from, final String subject,
		final String message) throws AddressException, UnsupportedEncodingException,
		MessagingException
	{
		String messageId = null;
		final Address[] fromAddress = EmailUtils.getAddressArray(EmailUtils.newAddress(from, null));
		final Address toAddress = EmailUtils.newAddress(to, null, null);
		final EmailMessage email = new EmailMessage();
		email.addFrom(fromAddress);
		email.addTo(toAddress);
		email.setSubject(subject);
		email.setContent(message, EmailConstants.MIMETYPE_TEXT_PLAIN);
		messageId = sendEmailMessage(email);
		return messageId;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setDebug(final boolean debug)
	{
		this.debug = debug;
		getSession().setDebug(debug);
	}

}
