/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.email.data.sources;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;

import javax.activation.DataSource;

import lombok.Getter;

/**
 * DataSource object for a byte array.
 */
public class ByteArrayDataSource implements DataSource
{

	/** The OutputStream for the Bytearray. */
	@Getter
	private final ByteArrayOutputStream outputStream;

	/** The contenttype. */
	@Getter
	private final String contentType;

	/**
	 * Constructor with the bytearray and the contenttype.
	 *
	 * @param byteArray
	 *            the bytearray.
	 * @param contenttype
	 *            the contenttype.
	 */
	public ByteArrayDataSource(final byte[] byteArray, final String contenttype)
	{
		this.contentType = contenttype;
		final int len = byteArray.length;
		outputStream = new ByteArrayOutputStream(len);
		outputStream.write(byteArray, 0, len);
	}

	/**
	 * Gets the byte array.
	 *
	 * @return Returns the data.
	 */
	public byte[] getByteArray()
	{
		return outputStream.toByteArray();
	}

	/**
	 * {@inheritDoc}
	 *
	 * @return the input stream
	 *
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 *
	 * @see javax.activation.DataSource#getInputStream()
	 */
	@Override
	public InputStream getInputStream() throws IOException
	{

		return new ByteArrayInputStream(outputStream.toByteArray());
	}

	/**
	 * {@inheritDoc}
	 *
	 * @return the name
	 *
	 * @see javax.activation.DataSource#getName()
	 */
	@Override
	public String getName()
	{
		return this.getClass().getCanonicalName();
	}

}
