/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.designpattern.observer.exception.handlers;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import net.sourceforge.jaulp.designpattern.observer.exception.ExceptionEvent;
import net.sourceforge.jaulp.designpattern.observer.exception.ExceptionListener;

/**
 * The abstract class that handles all exceptions.
 *
 * @version 1.0
 * @author Asterios Raptis
 */
public abstract class AbstractExceptionHandler implements Serializable
{

	/**
	 * The serialVersionUID.
	 */
	private static final long serialVersionUID = 1L;
	/** Lists of the ExceptionListeners. */
	private final List<ExceptionListener> listeners = new ArrayList<>();


	/**
	 * Adds an ExceptionListener object.
	 *
	 * @param listener
	 *            the listener
	 */
	void addExceptionListener(ExceptionListener listener)
	{
		listeners.add(listener);
	}

	/**
	 * Removes an ExceptionListener object.
	 *
	 * @param listener
	 *            the listener
	 */
	void removeExceptionListener(ExceptionListener listener)
	{
		listeners.remove(listener);
	}

	/**
	 * Update the given event.
	 *
	 * @param event
	 *            the event
	 */
	void update(ExceptionEvent event)
	{
		for (ExceptionListener listener : listeners)
		{
			updateEvent(listener, event);
		}
	}

	/**
	 * This method updates the listener for this event.
	 *
	 * @param listener
	 *            the listener
	 * @param event
	 *            the event
	 */
	public abstract void updateEvent(ExceptionListener listener, ExceptionEvent event);


}
