/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.designpattern.observer.exception;

import java.util.ArrayList;
import java.util.List;

/**
 * The Class ExceptionObservers.
 */
public class ExceptionObservers
{

	/** The instance. */
	private static ExceptionObservers instance = null;

	/**
	 * Gets the single instance of ExceptionObservers.
	 *
	 * @return single instance of ExceptionObservers
	 */
	public static synchronized ExceptionObservers getInstance()
	{
		synchronized (ExceptionObservers.class)
		{
			if (instance == null)
			{
				instance = new ExceptionObservers();
			}
		}
		return instance;
	}

	/** The exception listeners. */
	protected List<ExceptionListener> exceptionListeners = new ArrayList<>();

	/**
	 * Instantiates a new exception observers.
	 */
	private ExceptionObservers()
	{
	}

	/**
	 * Adds the exception listener. This method allows classes to register for ExceptionEvents.
	 *
	 * @param listener
	 *            the listener
	 */
	public void addExceptionListener(final ExceptionListener listener)
	{
		exceptionListeners.add(listener);
	}

	/**
	 * Fire exception event.
	 *
	 * @param event
	 *            the event
	 */
	void fireExceptionEvent(final ExceptionEvent event)
	{
		final int listenerSize = exceptionListeners.size();
		for (int i = 0; i < listenerSize; i++)
		{
			final ExceptionListener lis = exceptionListeners.get(i);
			lis.onException(event);
		}
	}


	/**
	 * Removes the exception listener. This method allows classes to unregister for ExceptionEvents.
	 *
	 * @param listener
	 *            the listener
	 */
	public void removeExceptionListener(final ExceptionListener listener)
	{
		exceptionListeners.remove(listener);
	}
}
