/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.designpattern.observer.event;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * The Class EventSubject.
 *
 * @param <T>
 *            the generic type
 */
public class EventSubject<T> implements EventSource<T>
{

	/**
	 * Initialize block.
	 **/
	{
		eventListeners = new ArrayList<EventListener<T>>();
	}

	/** The event listeners. */
	private List<EventListener<T>> eventListeners;

	/** The source. */
	private T source;

	/**
	 * Instantiates a new event subject.
	 */
	public EventSubject()
	{
		super();
	}

	/**
	 * Instantiates a new event subject.
	 *
	 * @param source
	 *            the source
	 */
	public EventSubject(final T source)
	{
		super();
		this.source = source;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void addEventListener(final EventListener<T> eventListener)
	{
		eventListeners.add(eventListener);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void addEventListeners(final Collection<EventListener<T>> eventListeners)
	{
		this.eventListeners.addAll(eventListeners);
	}

	/**
	 * Fires the source event and inform all registered listeners.
	 */
	private synchronized void fireEvent()
	{
		for (EventListener<T> eventListener : eventListeners)
		{
			eventListener.onEvent(source);
		}
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void fireEvent(final T source)
	{
		this.source = source;
		fireEvent();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void removeEventListener(final EventListener<T> eventListener)
	{
		eventListeners.remove(eventListener);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void removeEventListeners(final Collection<EventListener<T>> eventListeners)
	{
		this.eventListeners.removeAll(eventListeners);
	}
}
