/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.designpattern.observer.chat;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.sourceforge.jaulp.designpattern.observer.AbstractSubject;
import net.sourceforge.jaulp.designpattern.observer.ifaces.Subject;

/**
 * The Class ChatRoom.
 *
 * @param <M>
 *            the generic type of the Message that will be send in this chat room.
 */
public class ChatRoom<M extends Message<?>> extends AbstractSubject<M, ChatRoomUser<M>>
	implements
		Subject<M, ChatRoomUser<M>>,
		Room<M>,
		Serializable
{

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * Initialize block.
	 **/
	{
		observers = new ArrayList<>();
	}

	/** The observers. */
	private final List<ChatRoomUser<M>> observers;

	private final List<M> messageHistory = new ArrayList<>();

	/** The observable object. */
	private M observable;

	/** The name of the chat room. */
	private final String name;

	/**
	 * Gets the name of the chat room.
	 *
	 * @return the name of the chat room.
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * Constructor for a new chat room with the given name.
	 *
	 * @param name
	 *            the name of the chat room
	 */
	public ChatRoom(final String name)
	{
		this.name = name;
	}

	/**
	 * Constructor for a new subject with an observable.
	 *
	 * @param observable
	 *            the observable
	 * @param name
	 *            the name
	 */
	public ChatRoom(final M observable, String name)
	{
		this.observable = observable;
		this.name = name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void addObserver(final ChatRoomUser<M> observer)
	{
		observers.add(observer);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized M getObservable()
	{
		return observable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void removeObserver(final ChatRoomUser<M> observer)
	{
		int index = this.observers.indexOf(observer);
		if (0 <= index)
		{
			this.observers.remove(observer);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setObservable(final M observable)
	{
		this.observable = observable;
		messageHistory.add(observable);
		updateObservers();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void updateObservers()
	{
		for (ChatRoomUser<M> observer : this.observers)
		{
			observer.update(observable);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void addObservers(Collection<ChatRoomUser<M>> observers)
	{
		for (ChatRoomUser<M> chatUser : observers)
		{
			addObserver(chatUser);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void removeObservers(Collection<ChatRoomUser<M>> observers)
	{
		for (ChatRoomUser<M> chatUser : observers)
		{
			removeObserver(chatUser);
		}
	}

	/**
	 * Returns the number of chat users in this chat room.
	 *
	 * @return the number of chat users in this chat room.
	 */
	public int size()
	{
		return this.observers.size();
	}

	/**
	 * Gets the chat room users as User objects.
	 *
	 * @return the chat room users
	 */
	@Override
	public List<IUser<?>> getChatRoomUsers()
	{
		List<IUser<?>> chatRoomUsers = new ArrayList<>();
		for (ChatRoomUser<M> chatUser : observers)
		{
			chatRoomUsers.add(chatUser.getUser());
		}
		return chatRoomUsers;
	}

	@Override
	public boolean isSecure()
	{
		return false;
	}

	@Override
	public List<M> getMessageHistory()
	{
		return messageHistory;
	}

}
