/**
 * Copyright (C) 2007 Asterios Raptis
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.sourceforge.jaulp.designpattern.observer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import net.sourceforge.jaulp.designpattern.observer.ifaces.Observer;
import net.sourceforge.jaulp.designpattern.observer.ifaces.Subject;

/**
 * The Class AbstractSubject is an implementation from the interface Subject. This class
 * encapsulates the observable and fires an update if the observable changes. The update informs all
 * registered observers about the change of the observable.
 *
 * @param <T>
 *            the generic type of the observable.
 * @param <O>
 *            the generic type of the observer
 */
public abstract class AbstractSubject<T, O extends Observer<T>> implements Subject<T, O>
{

	/**
	 * Initialize block.
	 **/
	{
		observers = new ArrayList<>();
	}

	/** The observers. */
	private final List<O> observers;

	/** The observable object. */
	private T observable;

	/**
	 * Default constructor for a new subject.
	 */
	public AbstractSubject()
	{
		super();
	}

	/**
	 * Constructor for a new subject with an observable.
	 *
	 * @param observable
	 *            the observable
	 */
	public AbstractSubject(final T observable)
	{
		super();
		this.observable = observable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void addObserver(final O observer)
	{
		observers.add(observer);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void addObservers(final Collection<O> observers)
	{
		this.observers.addAll(observers);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized T getObservable()
	{
		return observable;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void removeObserver(final O observer)
	{
		int index = this.observers.indexOf(observer);
		if (0 <= index)
		{
			this.observers.remove(observer);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void removeObservers(final Collection<O> observers)
	{
		this.observers.removeAll(observers);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void setObservable(final T observable)
	{
		this.observable = observable;
		updateObservers();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void updateObservers()
	{
		for (O observer1 : this.observers)
		{
			Observer<T> observer = observer1;
			observer.update(observable);
		}
	}

}
