package de.aipark.api.parkingarea;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import de.aipark.api.chargingstation.ChargingPlugType;
import de.aipark.api.datasource.PublicDataSource;
import de.aipark.api.payment.PriceModel;
import io.swagger.annotations.ApiModelProperty;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;

import java.util.*;

/**
 * Created by torgen on 24.08.17.
 */
@SuppressWarnings({"WeakerAccess", "unused"})
@JsonIgnoreProperties(ignoreUnknown = true)
public class ParkingArea {
    @ApiModelProperty(value = "name of parking area", dataType = "java.lang.String", required = true, example = "Rathaus")
    private String name;

    @ApiModelProperty(value = "shape of parking area, e.g. multipolygon for areas or line for on street parking", required = true)
    private Geometry shape;

    @ApiModelProperty(value = "center point of parking area", required = true)
    private Point center;

    @ApiModelProperty(value = "capacity of parking area (number of spots)", dataType = "java.lang.Integer", required = true, example = "100")
    private Integer capacity;

    @ApiModelProperty(value = "type of parking area, types getting added in the future", dataType = "de.aipark.api.parkingarea.ParkingAreaType", required = true, example = "ON_STREET")
    private String parkingAreaType;

    @ApiModelProperty(value = "orientation of parking", dataType = "de.aipark.api.parkingarea.ParkingOrientation", example = "PARALLEL")
    private ParkingOrientation parkingOrientation;

    @ApiModelProperty(value = "unique id of parking area", dataType = "java.lang.Long", required = true, example = "504838")
    private Long id;

    @ApiModelProperty(value =
            "value=null indicates no information about capacity designated for disabled, " +
                    "value>0 indicates number of spots designated for women, " +
                    "value=0 indicates no special capacity for women, " +
                    "value=-1 indicates that there are spots designated for women but capacity is unknown",
            dataType = "java.lang.Integer", example = "10")
    private Integer capacityWoman;

    @ApiModelProperty(value =
            "value=null indicates no information about capacity designated for disabled, " +
                    "value>0 indicates number of spots designated for disabled, " +
                    "value=0 indicates no special capacity for disabled, " +
                    "value=-1 indicates that there are spots designated for disabled but capacity is unknown",
            dataType = "java.lang.Integer", example = "10")
    private Integer capacityDisabled;

    @ApiModelProperty(value = "schedule when parking area is opened or closed", dataType = "MapEntry<Schedule, OpeningStatus>")
    private MapEntry<Schedule, OpeningStatus> scheduleOpen;

    @ApiModelProperty(value = "schedule for parking disc (value = allowed time in minutes)", dataType = "MapEntry<Schedule, Integer>")
    private MapEntry<Schedule, Integer> scheduleMaxStay;

    @ApiModelProperty(value = "price model mapping", dataType = "MapEntry<Schedule, PriceModel>")
    private MapEntry<Schedule, PriceModel> schedulePriceModel;

    @ApiModelProperty(value = "schedule with capacity of parking area (number of spots) for residential parking only", dataType = "MapEntry<Schedule, Integer>")
    private MapEntry<Schedule, Integer> scheduleResidential;

    @ApiModelProperty(value = "entrance point for parking area", dataType = "List<EntrancePoint>")
    private List<EntrancePoint> entrancePoints;

    /**
     * will be removed in future release. Use special designated charging station methods.
     */
    @JsonIgnore
    @Deprecated
    @ApiModelProperty(hidden = true, value = "mapping of charging plug type to number of plugs with this type", dataType = "MapEntry<ChargingPlugType,Integer>")
    private MapEntry<ChargingPlugType, Integer> chargingPlugTypes; //TODO: remove

    @ApiModelProperty(value = "must publish datasource list", dataType = "List<PublicDataSource>")
    private List<PublicDataSource> mustPublishDatasourceList = new ArrayList<PublicDataSource>();

    public ParkingArea() {
        scheduleOpen = new MapEntry<Schedule, OpeningStatus>();
        scheduleMaxStay = new MapEntry<Schedule, Integer>();
        schedulePriceModel = new MapEntry<Schedule, PriceModel>();
        scheduleResidential = new MapEntry<Schedule, Integer>();
        entrancePoints = new ArrayList<EntrancePoint>();
        chargingPlugTypes = new MapEntry<ChargingPlugType, Integer>(); //TODO: remove
    }

//    /**
//     * use other constructor, this may be removed in future release
//     */
//    @Deprecated
//    public ParkingArea(String name, Geometry shape, Point center, Integer capacity, String parkingAreaType,
//                       Long id, Integer capacityWoman, Integer capacityDisabled, MapEntry<Schedule, OpeningStatus> scheduleOpen,
//                       MapEntry<Schedule, Integer> scheduleMaxStay, MapEntry<Schedule, PriceModel> schedulePriceModel, MapEntry<Schedule,
//            Integer> scheduleResidential, List<EntrancePoint> entrancePoints, MapEntry<ChargingPlugType, Integer> chargingPlugTypes) {
//        this(name, shape, center, capacity, parkingAreaType, null, id, capacityWoman, capacityDisabled, scheduleOpen,
//                scheduleMaxStay, schedulePriceModel, scheduleResidential, entrancePoints, chargingPlugTypes);
//    }

    public ParkingArea(String name, Geometry shape, Point center, Integer capacity, String parkingAreaType, ParkingOrientation parkingOrientation,
                       Long id, Integer capacityWoman, Integer capacityDisabled, MapEntry<Schedule, OpeningStatus> scheduleOpen,
                       MapEntry<Schedule, Integer> scheduleMaxStay, MapEntry<Schedule, PriceModel> schedulePriceModel, MapEntry<Schedule,
            Integer> scheduleResidential, List<EntrancePoint> entrancePoints, MapEntry<ChargingPlugType, Integer> chargingPlugTypes) {
        this.name = name;
        this.shape = shape;
        this.center = center;
        this.capacity = capacity;
        this.parkingAreaType = parkingAreaType;
        this.parkingOrientation = parkingOrientation;
        this.id = id;
        this.capacityWoman = capacityWoman;
        this.capacityDisabled = capacityDisabled;
        this.scheduleOpen = scheduleOpen;
        this.scheduleMaxStay = scheduleMaxStay;
        this.schedulePriceModel = schedulePriceModel;
        this.scheduleResidential = scheduleResidential;
        this.entrancePoints = entrancePoints;
//        this.chargingPlugTypes = chargingPlugTypes;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public Geometry getShape() {
        return shape;
    }

    public void setShape(Geometry shape) {
        this.shape = shape;
    }

    public Integer getCapacity() {
        return capacity;
    }

    public void setCapacity(Integer capacity) {
        this.capacity = capacity;
    }

    public ParkingAreaType getParkingAreaType() {
        try {
            return ParkingAreaType.valueOf(parkingAreaType);
        } catch (IllegalArgumentException e) {
            return ParkingAreaType.PARKINGAREA;
        }
    }

    public void setParkingAreaType(String parkingAreaType) {
        this.parkingAreaType = parkingAreaType;
    }

    public ParkingOrientation getParkingOrientation() {
        return parkingOrientation;
    }

    public void setParkingOrientation(ParkingOrientation parkingOrientation) {
        this.parkingOrientation = parkingOrientation;
    }

    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public Integer getCapacityWoman() {
        return capacityWoman;
    }

    public void setCapacityWoman(Integer capacityWoman) {
        this.capacityWoman = capacityWoman;
    }

    public Integer getCapacityDisabled() {
        return capacityDisabled;
    }

    public void setCapacityDisabled(Integer capacityDisabled) {
        this.capacityDisabled = capacityDisabled;
    }

    public MapEntry<Schedule, OpeningStatus> getScheduleOpen() {
        return scheduleOpen;
    }

    public void setScheduleOpen(MapEntry<Schedule, OpeningStatus> scheduleOpen) {
        this.scheduleOpen = scheduleOpen;
    }

    public MapEntry<Schedule, Integer> getScheduleMaxStay() {
        return scheduleMaxStay;
    }

    public void setScheduleMaxStay(MapEntry<Schedule, Integer> scheduleMaxStay) {
        this.scheduleMaxStay = scheduleMaxStay;
    }

    public MapEntry<Schedule, PriceModel> getSchedulePriceModel() {
        return schedulePriceModel;
    }

    public void setSchedulePriceModel(MapEntry<Schedule, PriceModel> schedulePriceModel) {
        this.schedulePriceModel = schedulePriceModel;
    }

    public MapEntry<Schedule, Integer> getScheduleResidential() {
        return scheduleResidential;
    }

    public void setScheduleResidential(MapEntry<Schedule, Integer> scheduleResidential) {
        this.scheduleResidential = scheduleResidential;
    }

    public List<EntrancePoint> getEntrancePoints() {
        return entrancePoints;
    }

    public void setEntrancePoints(List<EntrancePoint> entrancePoints) {
        this.entrancePoints = entrancePoints;
    }

    /**
     * will be removed in future release. Use special designated charging station methods.
     */
//    //@JsonIgnore
    @Deprecated
    public MapEntry<ChargingPlugType, Integer> getChargingPlugTypes() { //TODO: remove
        return chargingPlugTypes;
    }

    /**
     * will be removed in future release. Use special designated charging station methods.
     */
    @Deprecated
    public void setChargingPlugTypes(MapEntry<ChargingPlugType, Integer> chargingPlugTypes) { //TODO: remove
        this.chargingPlugTypes = chargingPlugTypes;
    }

    public Point getCenter() {
        return center;
    }

    public void setCenter(Point center) {
        this.center = center;
    }

    public List<PublicDataSource> getMustPublishDatasourceList() {
        return mustPublishDatasourceList;
    }

    public void setMustPublishDatasourceList(List<PublicDataSource> mustPublishDatasourceList) {
        this.mustPublishDatasourceList = mustPublishDatasourceList;
    }

    @Override
    public String toString() {
        return "ParkingArea{" +
                "name='" + name + '\'' +
                ", shape=" + shape +
                ", center=" + center +
                ", capacity=" + capacity +
                ", parkingAreaType='" + parkingAreaType + '\'' +
                ", parkingOrientation=" + parkingOrientation +
                ", id=" + id +
                ", capacityWoman=" + capacityWoman +
                ", capacityDisabled=" + capacityDisabled +
                ", scheduleOpen=" + scheduleOpen +
                ", scheduleMaxStay=" + scheduleMaxStay +
                ", schedulePriceModel=" + schedulePriceModel +
                ", scheduleResidential=" + scheduleResidential +
                ", entrancePoints=" + entrancePoints +
//                ", chargingPlugTypes=" + chargingPlugTypes +
                '}';
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof ParkingArea)) return false;

        ParkingArea that = (ParkingArea) o;

        if (name != null ? !name.equals(that.name) : that.name != null) return false;
        if (shape != null ? !shape.equals(that.shape) : that.shape != null) return false;
        if (center != null ? !center.equals(that.center) : that.center != null) return false;
        if (capacity != null ? !capacity.equals(that.capacity) : that.capacity != null) return false;
        if (parkingAreaType != null ? !parkingAreaType.equals(that.parkingAreaType) : that.parkingAreaType != null)
            return false;
        if (parkingOrientation != that.parkingOrientation) return false;
        if (id != null ? !id.equals(that.id) : that.id != null) return false;
        if (capacityWoman != null ? !capacityWoman.equals(that.capacityWoman) : that.capacityWoman != null)
            return false;
        if (capacityDisabled != null ? !capacityDisabled.equals(that.capacityDisabled) : that.capacityDisabled != null)
            return false;
        if (scheduleOpen != null ? !scheduleOpen.equals(that.scheduleOpen) : that.scheduleOpen != null) return false;
        if (scheduleMaxStay != null ? !scheduleMaxStay.equals(that.scheduleMaxStay) : that.scheduleMaxStay != null)
            return false;
        if (schedulePriceModel != null ? !schedulePriceModel.equals(that.schedulePriceModel) : that.schedulePriceModel != null)
            return false;
        if (scheduleResidential != null ? !scheduleResidential.equals(that.scheduleResidential) : that.scheduleResidential != null)
            return false;
        if (entrancePoints != null ? !entrancePoints.equals(that.entrancePoints) : that.entrancePoints != null)
            return false;
        if (chargingPlugTypes != null ? !chargingPlugTypes.equals(that.chargingPlugTypes) : that.chargingPlugTypes != null)
            return false;
        return mustPublishDatasourceList != null ? mustPublishDatasourceList.equals(that.mustPublishDatasourceList) : that.mustPublishDatasourceList == null;
    }

    @Override
    public int hashCode() {
        int result = name != null ? name.hashCode() : 0;
        result = 31 * result + (shape != null ? shape.hashCode() : 0);
        result = 31 * result + (center != null ? center.hashCode() : 0);
        result = 31 * result + (capacity != null ? capacity.hashCode() : 0);
        result = 31 * result + (parkingAreaType != null ? parkingAreaType.hashCode() : 0);
        result = 31 * result + (parkingOrientation != null ? parkingOrientation.hashCode() : 0);
        result = 31 * result + (id != null ? id.hashCode() : 0);
        result = 31 * result + (capacityWoman != null ? capacityWoman.hashCode() : 0);
        result = 31 * result + (capacityDisabled != null ? capacityDisabled.hashCode() : 0);
        result = 31 * result + (scheduleOpen != null ? scheduleOpen.hashCode() : 0);
        result = 31 * result + (scheduleMaxStay != null ? scheduleMaxStay.hashCode() : 0);
        result = 31 * result + (schedulePriceModel != null ? schedulePriceModel.hashCode() : 0);
        result = 31 * result + (scheduleResidential != null ? scheduleResidential.hashCode() : 0);
        result = 31 * result + (entrancePoints != null ? entrancePoints.hashCode() : 0);
        result = 31 * result + (chargingPlugTypes != null ? chargingPlugTypes.hashCode() : 0);
        result = 31 * result + (mustPublishDatasourceList != null ? mustPublishDatasourceList.hashCode() : 0);
        return result;
    }

    @SuppressWarnings("StringConcatenationInLoop") //compatibility with js converter
    @JsonIgnore
    public String getPriceString() {
        ResourceBundle bundle = ResourceBundle.getBundle("Api");
        String result = "";
        Map<Schedule, List<PriceModel>> scheduleListMap = new HashMap<Schedule, List<PriceModel>>();
        for (MapEntry.Entry<Schedule, PriceModel> entry : schedulePriceModel.getEntryList()) {
            List<PriceModel> priceModels = scheduleListMap.get(entry.getKey());
            if (priceModels == null) {
                priceModels = new ArrayList<PriceModel>();
            }
            priceModels.add(entry.getValue());
            scheduleListMap.put(entry.getKey(), priceModels);
        }
        List<Schedule> schedules = new ArrayList<Schedule>(scheduleListMap.keySet());
        Collections.sort(schedules);
        boolean onlyDefaultPriceModels = false;
        for (Schedule schedule : schedules) {
            if (schedule.toString().isEmpty()) {
                if (schedules.size() > 1) {
                    result += bundle.getString("otherwise") + ":\n";
                } else {
                    onlyDefaultPriceModels = true;
                }
            } else {
                result += schedule.toString() + ":\n";
            }
            List<PriceModel> priceModels = new ArrayList<PriceModel>(scheduleListMap.get(schedule));
            Collections.sort(priceModels);
            boolean hasPeriodPricemodels = false;
            for (PriceModel priceModel : priceModels) {
                if (!onlyDefaultPriceModels) {
                    result += "\t";
                }
                if (priceModel.getFromMinute() != null || priceModel.getUntilMinute() != null) {
                    hasPeriodPricemodels = true;
                } else if (hasPeriodPricemodels && priceModel.getInterval() != null && priceModel.getMaxPriceInCent() == null) {
                    result += bundle.getString("otherwise") + " ";
                }
                result += priceModel.toString() + "\n";
            }
        }
        return result;
    }

    @SuppressWarnings("StringConcatenationInLoop") //compatibility with js converter
    @JsonIgnore
    public String getMaxStayString() {
        ResourceBundle bundle = ResourceBundle.getBundle("Api");
        String result = "";
        for (MapEntry.Entry<Schedule, Integer> entry : scheduleMaxStay.getEntryList()) {
            if (!result.isEmpty()) {
                result += "\n";
            }
            result += entry.getKey().toString() + " " + bundle.getString("maximal") + " " + entry.getValue() + " " + bundle.getString("minutes");
        }
        return result;
    }

    @SuppressWarnings("StringConcatenationInLoop") //compatibility with js converter
    @JsonIgnore
    public String getOpeningHoursString() {
        ResourceBundle bundle = ResourceBundle.getBundle("Api");
        String result = "";
        for(MapEntry.Entry<Schedule,OpeningStatus> entry :scheduleOpen.getEntryList()){
            String opened = bundle.getString("closed");
            if(entry.getValue().equals(OpeningStatus.OPEN)){
                    opened = bundle.getString("open");
            }
            result += entry.getKey().toString() + " " + opened + "\n";
        }
        if(result.length() > 0){
            result = result.substring(0, result.length() - 1);
        }
        return result;
    }
}
