/*
 * Decompiled with CFR 0.152.
 */
package de.adorsys.sts.keymanagement.service;

import de.adorsys.sts.keymanagement.config.KeyManagementRotationProperties;
import de.adorsys.sts.keymanagement.model.KeyUsage;
import de.adorsys.sts.keymanagement.model.StsKeyEntry;
import de.adorsys.sts.keymanagement.model.StsKeyStore;
import de.adorsys.sts.keymanagement.service.KeyStoreGenerator;
import de.adorsys.sts.keymanagement.util.DateTimeUtils;
import java.time.Clock;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

public class KeyRotationService {
    private final KeyStoreGenerator keyStoreGenerator;
    private final Clock clock;
    private final KeyManagementRotationProperties.KeyRotationProperties encryptionKeyPairRotationProperties;
    private final KeyManagementRotationProperties.KeyRotationProperties signatureKeyPairRotationProperties;
    private final KeyManagementRotationProperties.KeyRotationProperties secretKeyRotationProperties;

    public KeyRotationService(KeyStoreGenerator keyStoreGenerator, Clock clock, KeyManagementRotationProperties rotationProperties) {
        this.keyStoreGenerator = keyStoreGenerator;
        this.clock = clock;
        this.encryptionKeyPairRotationProperties = rotationProperties.getEncKeyPairs();
        this.signatureKeyPairRotationProperties = rotationProperties.getSignKeyPairs();
        this.secretKeyRotationProperties = rotationProperties.getSecretKeys();
    }

    public KeyRotationResult rotate(StsKeyStore stsKeyStore) {
        KeyStateUpdates keyStateUpdates = this.updateKeyStates(stsKeyStore);
        List<String> createdFutureKeys = this.createKeysForFutureUsage(stsKeyStore, keyStateUpdates);
        List<String> removedKeys = this.removeExpiredKeys(stsKeyStore);
        List<String> generatedKeyAliases = this.generateAndAddMissingKeys(stsKeyStore);
        return KeyRotationResult.builder().generatedKeys(generatedKeyAliases).removedKeys(removedKeys).futureKeys(createdFutureKeys).build();
    }

    private List<String> createKeysForFutureUsage(StsKeyStore stsKeyStore, KeyStateUpdates keyStateUpdates) {
        ArrayList<String> generatedKeyAliases = new ArrayList<String>();
        for (StsKeyEntry keyEntry : keyStateUpdates.newValidKeys) {
            StsKeyEntry generatedKeyEntry = this.keyStoreGenerator.generateKeyEntryForFutureUsage(keyEntry.getKeyUsage(), keyEntry.getNotAfter());
            stsKeyStore.addKey(generatedKeyEntry);
            generatedKeyAliases.add(generatedKeyEntry.getAlias());
        }
        return generatedKeyAliases;
    }

    private KeyStateUpdates updateKeyStates(StsKeyStore stsKeyStore) {
        KeyStateUpdates keyStateUpdates;
        KeyStateUpdates updates = new KeyStateUpdates();
        ZonedDateTime now = this.now();
        if (this.encryptionKeyPairRotationProperties.isEnabled().booleanValue()) {
            keyStateUpdates = this.updateEncryptionKeyEntryStates(stsKeyStore, now);
            updates.merge(keyStateUpdates);
        }
        if (this.signatureKeyPairRotationProperties.isEnabled().booleanValue()) {
            keyStateUpdates = this.updateSignatureKeyEntryStates(stsKeyStore, now);
            updates.merge(keyStateUpdates);
        }
        if (this.secretKeyRotationProperties.isEnabled().booleanValue()) {
            keyStateUpdates = this.updateSecretKeyEntryStates(stsKeyStore, now);
            updates.merge(keyStateUpdates);
        }
        return updates;
    }

    private KeyStateUpdates updateEncryptionKeyEntryStates(StsKeyStore stsKeyStore, ZonedDateTime now) {
        List<StsKeyEntry> encryptionKeyEntries = stsKeyStore.getKeyEntries().values().stream().filter(k -> k.getKeyUsage() == KeyUsage.Encryption).collect(Collectors.toList());
        return this.updateKeyEntryStatesForCollection(now, encryptionKeyEntries);
    }

    private KeyStateUpdates updateSignatureKeyEntryStates(StsKeyStore stsKeyStore, ZonedDateTime now) {
        List<StsKeyEntry> encryptionKeyEntries = stsKeyStore.getKeyEntries().values().stream().filter(k -> k.getKeyUsage() == KeyUsage.Signature).collect(Collectors.toList());
        return this.updateKeyEntryStatesForCollection(now, encryptionKeyEntries);
    }

    private KeyStateUpdates updateSecretKeyEntryStates(StsKeyStore stsKeyStore, ZonedDateTime now) {
        List<StsKeyEntry> encryptionKeyEntries = stsKeyStore.getKeyEntries().values().stream().filter(k -> k.getKeyUsage() == KeyUsage.SecretKey).collect(Collectors.toList());
        return this.updateKeyEntryStatesForCollection(now, encryptionKeyEntries);
    }

    private KeyStateUpdates updateKeyEntryStatesForCollection(ZonedDateTime now, List<StsKeyEntry> encryptionKeyEntries) {
        KeyStateUpdates updates = new KeyStateUpdates();
        List keysToBeValid = encryptionKeyEntries.stream().filter(k -> k.getState() == StsKeyEntry.State.CREATED).filter(k -> k.getNotBefore().isBefore(now)).collect(Collectors.toList());
        for (Object keyEntry : keysToBeValid) {
            ZonedDateTime notAfter = DateTimeUtils.addMillis(now, ((StsKeyEntry)keyEntry).getValidityInterval());
            ((StsKeyEntry)keyEntry).setNotAfter(notAfter);
            ((StsKeyEntry)keyEntry).setExpireAt(DateTimeUtils.addMillis(notAfter, ((StsKeyEntry)keyEntry).getValidityInterval()));
            ((StsKeyEntry)keyEntry).setState(StsKeyEntry.State.VALID);
        }
        updates.newValidKeys = keysToBeValid;
        List keysToBeLegacy = encryptionKeyEntries.stream().filter(k -> k.getState() == StsKeyEntry.State.VALID).filter(k -> now.isAfter(k.getNotAfter())).collect(Collectors.toList());
        for (StsKeyEntry keyEntry : keysToBeLegacy) {
            keyEntry.setState(StsKeyEntry.State.LEGACY);
        }
        updates.newLegacyKeys = keysToBeLegacy;
        List keysToBeExpired = encryptionKeyEntries.stream().filter(k -> k.getState() == StsKeyEntry.State.LEGACY).filter(k -> now.isAfter(k.getExpireAt())).collect(Collectors.toList());
        for (StsKeyEntry keyEntry : keysToBeExpired) {
            keyEntry.setState(StsKeyEntry.State.EXPIRED);
        }
        updates.newExpiredKeys = keysToBeExpired;
        return updates;
    }

    private List<String> removeExpiredKeys(StsKeyStore stsKeyStore) {
        ArrayList<String> removedKeys = new ArrayList<String>();
        if (this.encryptionKeyPairRotationProperties.isEnabled().booleanValue()) {
            removedKeys.addAll(this.removeExpiredKeys(stsKeyStore, KeyUsage.Encryption));
        }
        if (this.signatureKeyPairRotationProperties.isEnabled().booleanValue()) {
            removedKeys.addAll(this.removeExpiredKeys(stsKeyStore, KeyUsage.Signature));
        }
        if (this.secretKeyRotationProperties.isEnabled().booleanValue()) {
            removedKeys.addAll(this.removeExpiredKeys(stsKeyStore, KeyUsage.SecretKey));
        }
        return removedKeys;
    }

    private List<String> removeExpiredKeys(StsKeyStore stsKeyStore, KeyUsage keyUsage) {
        Collection<StsKeyEntry> actualKeys = stsKeyStore.getKeyEntries().values();
        ArrayList<StsKeyEntry> copiedKeyEntries = new ArrayList<StsKeyEntry>(actualKeys);
        return copiedKeyEntries.stream().filter(k -> k.getState() == StsKeyEntry.State.EXPIRED).filter(k -> k.getKeyUsage() == keyUsage).map(k -> this.removeKey(stsKeyStore, (StsKeyEntry)k)).collect(Collectors.toList());
    }

    private List<String> generateAndAddMissingKeys(StsKeyStore stsKeyStore) {
        Collection<StsKeyEntry> actualKeys = stsKeyStore.getKeyEntries().values();
        List<StsKeyEntry> generatedKeys = this.generateMissingKeys(actualKeys);
        ArrayList<String> generatedKeyAliases = new ArrayList<String>();
        for (StsKeyEntry generatedKey : generatedKeys) {
            stsKeyStore.addKey(generatedKey);
            generatedKeyAliases.add(generatedKey.getAlias());
        }
        return generatedKeyAliases;
    }

    private List<StsKeyEntry> generateMissingKeys(Collection<StsKeyEntry> actualKeys) {
        ArrayList<StsKeyEntry> generatedKeys = new ArrayList<StsKeyEntry>();
        if (this.encryptionKeyPairRotationProperties.isEnabled().booleanValue()) {
            generatedKeys.addAll(this.generateMissingEncryptionKeys(actualKeys));
        }
        if (this.signatureKeyPairRotationProperties.isEnabled().booleanValue()) {
            generatedKeys.addAll(this.generateMissingSignatureKeys(actualKeys));
        }
        if (this.secretKeyRotationProperties.isEnabled().booleanValue()) {
            generatedKeys.addAll(this.generateMissingSecretKeys(actualKeys));
        }
        return generatedKeys;
    }

    private List<StsKeyEntry> generateMissingEncryptionKeys(Collection<StsKeyEntry> actualKeys) {
        ArrayList<StsKeyEntry> generatedKeys = new ArrayList<StsKeyEntry>();
        long countOfValidEncryptionKeyPairs = actualKeys.stream().filter(k -> k.getState() == StsKeyEntry.State.VALID).filter(k -> k.getKeyUsage() == KeyUsage.Encryption).count();
        int i = 0;
        while ((long)i < (long)this.encryptionKeyPairRotationProperties.getMinKeys().intValue() - countOfValidEncryptionKeyPairs) {
            StsKeyEntry generatedKeyAlias = this.generateKey(KeyUsage.Encryption);
            generatedKeys.add(generatedKeyAlias);
            ++i;
        }
        return generatedKeys;
    }

    private List<StsKeyEntry> generateMissingSignatureKeys(Collection<StsKeyEntry> actualKeys) {
        ArrayList<StsKeyEntry> generatedKeys = new ArrayList<StsKeyEntry>();
        long countOfValidSignatureKeyPairs = actualKeys.stream().filter(k -> k.getState() == StsKeyEntry.State.VALID).filter(k -> k.getKeyUsage() == KeyUsage.Signature).count();
        int i = 0;
        while ((long)i < (long)this.signatureKeyPairRotationProperties.getMinKeys().intValue() - countOfValidSignatureKeyPairs) {
            StsKeyEntry generatedKeyAlias = this.generateKey(KeyUsage.Signature);
            generatedKeys.add(generatedKeyAlias);
            ++i;
        }
        return generatedKeys;
    }

    private List<StsKeyEntry> generateMissingSecretKeys(Collection<StsKeyEntry> actualKeys) {
        ArrayList<StsKeyEntry> generatedKeys = new ArrayList<StsKeyEntry>();
        long countOfValidSecretKeys = actualKeys.stream().filter(k -> k.getState() == StsKeyEntry.State.VALID).filter(k -> k.getKeyUsage() == KeyUsage.SecretKey).count();
        int i = 0;
        while ((long)i < (long)this.secretKeyRotationProperties.getMinKeys().intValue() - countOfValidSecretKeys) {
            StsKeyEntry generatedKeyAlias = this.generateKey(KeyUsage.SecretKey);
            generatedKeys.add(generatedKeyAlias);
            ++i;
        }
        return generatedKeys;
    }

    private String removeKey(StsKeyStore keyStore, StsKeyEntry stsKeyEntry) {
        String alias = stsKeyEntry.getAlias();
        keyStore.removeKey(alias);
        return alias;
    }

    private StsKeyEntry generateKey(KeyUsage keyUsage) {
        StsKeyEntry stsKeyEntry;
        if (keyUsage == KeyUsage.Signature) {
            stsKeyEntry = this.keyStoreGenerator.generateSignatureKeyEntryForInstantUsage();
        } else if (keyUsage == KeyUsage.Encryption) {
            stsKeyEntry = this.keyStoreGenerator.generateEncryptionKeyEntryForInstantUsage();
        } else if (keyUsage == KeyUsage.SecretKey) {
            stsKeyEntry = this.keyStoreGenerator.generateSecretKeyEntryForInstantUsage();
        } else {
            throw new IllegalArgumentException("Unknown KeyUsage: " + (Object)((Object)keyUsage));
        }
        return stsKeyEntry;
    }

    private ZonedDateTime now() {
        return this.clock.instant().atZone(ZoneOffset.UTC);
    }

    public static class KeyRotationResult {
        private List<String> removedKeys;
        private List<String> futureKeys;
        private List<String> generatedKeys;

        private static List<String> $default$removedKeys() {
            return new ArrayList<String>();
        }

        private static List<String> $default$futureKeys() {
            return new ArrayList<String>();
        }

        private static List<String> $default$generatedKeys() {
            return new ArrayList<String>();
        }

        KeyRotationResult(List<String> removedKeys, List<String> futureKeys, List<String> generatedKeys) {
            this.removedKeys = removedKeys;
            this.futureKeys = futureKeys;
            this.generatedKeys = generatedKeys;
        }

        public static KeyRotationResultBuilder builder() {
            return new KeyRotationResultBuilder();
        }

        public List<String> getRemovedKeys() {
            return this.removedKeys;
        }

        public List<String> getFutureKeys() {
            return this.futureKeys;
        }

        public List<String> getGeneratedKeys() {
            return this.generatedKeys;
        }

        public static class KeyRotationResultBuilder {
            private boolean removedKeys$set;
            private List<String> removedKeys;
            private boolean futureKeys$set;
            private List<String> futureKeys;
            private boolean generatedKeys$set;
            private List<String> generatedKeys;

            KeyRotationResultBuilder() {
            }

            public KeyRotationResultBuilder removedKeys(List<String> removedKeys) {
                this.removedKeys = removedKeys;
                this.removedKeys$set = true;
                return this;
            }

            public KeyRotationResultBuilder futureKeys(List<String> futureKeys) {
                this.futureKeys = futureKeys;
                this.futureKeys$set = true;
                return this;
            }

            public KeyRotationResultBuilder generatedKeys(List<String> generatedKeys) {
                this.generatedKeys = generatedKeys;
                this.generatedKeys$set = true;
                return this;
            }

            public KeyRotationResult build() {
                List removedKeys = this.removedKeys;
                if (!this.removedKeys$set) {
                    removedKeys = KeyRotationResult.$default$removedKeys();
                }
                List futureKeys = this.futureKeys;
                if (!this.futureKeys$set) {
                    futureKeys = KeyRotationResult.$default$futureKeys();
                }
                List generatedKeys = this.generatedKeys;
                if (!this.generatedKeys$set) {
                    generatedKeys = KeyRotationResult.$default$generatedKeys();
                }
                return new KeyRotationResult(removedKeys, futureKeys, generatedKeys);
            }

            public String toString() {
                return "KeyRotationService.KeyRotationResult.KeyRotationResultBuilder(removedKeys=" + this.removedKeys + ", futureKeys=" + this.futureKeys + ", generatedKeys=" + this.generatedKeys + ")";
            }
        }
    }

    private static class KeyStateUpdates {
        public List<StsKeyEntry> newValidKeys = new ArrayList<StsKeyEntry>();
        public List<StsKeyEntry> newLegacyKeys = new ArrayList<StsKeyEntry>();
        public List<StsKeyEntry> newExpiredKeys = new ArrayList<StsKeyEntry>();

        private KeyStateUpdates() {
        }

        public void merge(KeyStateUpdates other) {
            this.newValidKeys.addAll(other.newValidKeys);
            this.newLegacyKeys.addAll(other.newLegacyKeys);
            this.newExpiredKeys.addAll(other.newExpiredKeys);
        }
    }
}

