/*
 * Decompiled with CFR 0.152.
 */
package de.adorsys.sdjwt;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.ECDSAVerifier;
import com.nimbusds.jose.crypto.Ed25519Verifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.KeyType;
import de.adorsys.sdjwt.IssuerSignedJWT;
import de.adorsys.sdjwt.IssuerSignedJwtVerificationOpts;
import de.adorsys.sdjwt.SdJwtUtils;
import de.adorsys.sdjwt.VisibleSdJwtClaim;
import de.adorsys.sdjwt.exception.SdJwtVerificationException;
import de.adorsys.sdjwt.vp.KeyBindingJWT;
import de.adorsys.sdjwt.vp.KeyBindingJwtVerificationOpts;
import java.text.ParseException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

public class SdJwtVerificationContext {
    private String sdJwtVpString;
    private final IssuerSignedJWT issuerSignedJwt;
    private final Map<String, String> disclosures;
    private KeyBindingJWT keyBindingJwt;

    public SdJwtVerificationContext(String sdJwtVpString, IssuerSignedJWT issuerSignedJwt, Map<String, String> disclosures, KeyBindingJWT keyBindingJwt) {
        this(issuerSignedJwt, disclosures);
        this.keyBindingJwt = keyBindingJwt;
        this.sdJwtVpString = sdJwtVpString;
    }

    public SdJwtVerificationContext(IssuerSignedJWT issuerSignedJwt, Map<String, String> disclosures) {
        this.issuerSignedJwt = issuerSignedJwt;
        this.disclosures = disclosures;
    }

    public SdJwtVerificationContext(IssuerSignedJWT issuerSignedJwt, List<String> disclosureStrings) {
        this.issuerSignedJwt = issuerSignedJwt;
        this.disclosures = this.computeDigestDisclosureMap(disclosureStrings);
    }

    private Map<String, String> computeDigestDisclosureMap(List<String> disclosureStrings) {
        return disclosureStrings.stream().map(disclosureString -> {
            String digest = SdJwtUtils.hashAndBase64EncodeNoPad(disclosureString.getBytes(), this.issuerSignedJwt.getSdHashAlg());
            return Map.entry(digest, disclosureString);
        }).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    public void verifyIssuance(IssuerSignedJwtVerificationOpts issuerSignedJwtVerificationOpts) throws SdJwtVerificationException {
        this.validateIssuerSignedJwt(issuerSignedJwtVerificationOpts.getVerifier());
        JsonNode disclosedPayload = this.validateDisclosuresDigests();
        this.validateIssuerSignedJwtTimeClaims(disclosedPayload, issuerSignedJwtVerificationOpts);
    }

    public void verifyPresentation(IssuerSignedJwtVerificationOpts issuerSignedJwtVerificationOpts, KeyBindingJwtVerificationOpts keyBindingJwtVerificationOpts) throws SdJwtVerificationException {
        if (keyBindingJwtVerificationOpts.isKeyBindingRequired() && this.keyBindingJwt == null) {
            throw new SdJwtVerificationException("Missing Key Binding JWT");
        }
        this.verifyIssuance(issuerSignedJwtVerificationOpts);
        if (keyBindingJwtVerificationOpts.isKeyBindingRequired()) {
            this.validateKeyBindingJwt(keyBindingJwtVerificationOpts);
        }
    }

    private void validateIssuerSignedJwt(JWSVerifier verifier) throws SdJwtVerificationException {
        this.issuerSignedJwt.verifySdHashAlgorithm();
        try {
            this.issuerSignedJwt.verifySignature(verifier);
        }
        catch (JOSEException e) {
            throw new SdJwtVerificationException("Invalid Issuer-Signed JWT", e);
        }
    }

    private void validateKeyBindingJwt(KeyBindingJwtVerificationOpts keyBindingJwtVerificationOpts) throws SdJwtVerificationException {
        this.validateKeyBindingJwtTyp();
        JsonNode cnf = this.issuerSignedJwt.getCnfClaim().orElseThrow(() -> new SdJwtVerificationException("No cnf claim in Issuer-signed JWT for key binding"));
        JWSVerifier holderVerifier = this.buildHolderVerifier(cnf);
        try {
            this.keyBindingJwt.verifySignature(holderVerifier);
        }
        catch (JOSEException e) {
            throw new SdJwtVerificationException("Key binding JWT invalid", e);
        }
        this.validateKeyBindingJwtTimeClaims(keyBindingJwtVerificationOpts);
        this.preventKeyBindingJwtReplay(keyBindingJwtVerificationOpts);
        this.validateKeyBindingJwtSdHashIntegrity();
    }

    private void validateKeyBindingJwtTyp() throws SdJwtVerificationException {
        JsonNode typ = this.keyBindingJwt.getHeader().get("typ");
        if (typ == null || !typ.isTextual() || !typ.asText().equals("kb+jwt")) {
            throw new SdJwtVerificationException("Key Binding JWT is not of declared typ kb+jwt");
        }
    }

    private JWSVerifier buildHolderVerifier(JsonNode cnf) throws SdJwtVerificationException {
        RSASSAVerifier verifier;
        block8: {
            JWK jwk;
            Objects.requireNonNull(cnf);
            JsonNode cnfJwk = cnf.get("jwk");
            if (cnfJwk == null) {
                throw new UnsupportedOperationException("Only cnf/jwk claim supported");
            }
            try {
                jwk = JWK.parse((String)cnfJwk.toString());
            }
            catch (ParseException e) {
                throw new SdJwtVerificationException("Malformed cnf/jwk claim");
            }
            try {
                if (KeyType.RSA.equals((Object)jwk.getKeyType())) {
                    verifier = new RSASSAVerifier(jwk.toRSAKey());
                    break block8;
                }
                if (KeyType.EC.equals((Object)jwk.getKeyType())) {
                    verifier = new ECDSAVerifier(jwk.toECKey());
                    break block8;
                }
                if (KeyType.OKP.equals((Object)jwk.getKeyType())) {
                    verifier = new Ed25519Verifier(jwk.toOctetKeyPair());
                    break block8;
                }
                throw new SdJwtVerificationException("cnf/jwk alg is unsupported or deemed not secure");
            }
            catch (JOSEException e) {
                throw new SdJwtVerificationException("cnf/jwk is unsupported or invalid");
            }
        }
        return verifier;
    }

    private void validateIssuerSignedJwtTimeClaims(JsonNode payload, IssuerSignedJwtVerificationOpts issuerSignedJwtVerificationOpts) throws SdJwtVerificationException {
        long now = Instant.now().getEpochSecond();
        try {
            if (issuerSignedJwtVerificationOpts.mustValidateIssuedAtClaim() && now < SdJwtUtils.readTimeClaim(payload, "iat")) {
                throw new SdJwtVerificationException("JWT issued in the future");
            }
        }
        catch (SdJwtVerificationException e) {
            throw new SdJwtVerificationException("Issuer-Signed JWT: Invalid `iat` claim", e);
        }
        try {
            if (issuerSignedJwtVerificationOpts.mustValidateExpirationClaim() && now >= SdJwtUtils.readTimeClaim(payload, "exp")) {
                throw new SdJwtVerificationException("JWT has expired");
            }
        }
        catch (SdJwtVerificationException e) {
            throw new SdJwtVerificationException("Issuer-Signed JWT: Invalid `exp` claim", e);
        }
        try {
            if (issuerSignedJwtVerificationOpts.mustValidateNotBeforeClaim() && now < SdJwtUtils.readTimeClaim(payload, "nbf")) {
                throw new SdJwtVerificationException("JWT is not yet valid");
            }
        }
        catch (SdJwtVerificationException e) {
            throw new SdJwtVerificationException("Issuer-Signed JWT: Invalid `nbf` claim", e);
        }
    }

    private void validateKeyBindingJwtTimeClaims(KeyBindingJwtVerificationOpts keyBindingJwtVerificationOpts) throws SdJwtVerificationException {
        try {
            this.keyBindingJwt.verifyIssuedAtClaim();
        }
        catch (SdJwtVerificationException e) {
            throw new SdJwtVerificationException("Key binding JWT: Invalid `iat` claim", e);
        }
        long now = Instant.now().getEpochSecond();
        long keyBindingJwtIat = SdJwtUtils.readTimeClaim(this.keyBindingJwt.getPayload(), "iat");
        if (now - keyBindingJwtIat > (long)keyBindingJwtVerificationOpts.getAllowedMaxAge()) {
            throw new SdJwtVerificationException("Key binding JWT is too old");
        }
        try {
            if (keyBindingJwtVerificationOpts.mustValidateExpirationClaim()) {
                this.keyBindingJwt.verifyExpClaim();
            }
        }
        catch (SdJwtVerificationException e) {
            throw new SdJwtVerificationException("Key binding JWT: Invalid `exp` claim", e);
        }
        try {
            if (keyBindingJwtVerificationOpts.mustValidateNotBeforeClaim()) {
                this.keyBindingJwt.verifyNotBeforeClaim();
            }
        }
        catch (SdJwtVerificationException e) {
            throw new SdJwtVerificationException("Key binding JWT: Invalid `nbf` claim", e);
        }
    }

    private JsonNode validateDisclosuresDigests() throws SdJwtVerificationException {
        HashSet<String> visitedDigests = new HashSet<String>();
        HashSet<String> visitedDisclosureStrings = new HashSet<String>();
        JsonNode disclosedPayload = this.validateViaRecursiveDisclosing(this.issuerSignedJwt.getPayload(), visitedDigests, visitedDisclosureStrings);
        this.validateDisclosuresVisits(visitedDisclosureStrings);
        return disclosedPayload;
    }

    private JsonNode validateViaRecursiveDisclosing(JsonNode currentNode, Set<String> visitedDigests, Set<String> visitedDisclosureStrings) throws SdJwtVerificationException {
        if (!currentNode.isObject() && !currentNode.isArray()) {
            return currentNode;
        }
        if (currentNode.isObject()) {
            ObjectNode currentObjectNode = (ObjectNode)currentNode;
            JsonNode sdArray = currentObjectNode.get("_sd");
            if (sdArray != null && sdArray.isArray()) {
                for (JsonNode el : sdArray) {
                    if (!el.isTextual()) {
                        throw new SdJwtVerificationException("Unexpected non-string element inside _sd array: " + el);
                    }
                    String digest = el.asText();
                    this.markDigestAsVisited(digest, visitedDigests);
                    String disclosure = this.disclosures.get(digest);
                    if (disclosure == null) continue;
                    visitedDisclosureStrings.add(disclosure);
                    VisibleSdJwtClaim claim = this.validateSdArrayDigestDisclosureFormat(disclosure);
                    currentObjectNode.set(claim.getClaimNameAsString(), claim.getVisibleClaimValue(null));
                }
            }
            currentObjectNode.remove("_sd");
            currentObjectNode.remove("_sd_alg");
        }
        if (currentNode.isArray()) {
            ArrayNode currentArrayNode = (ArrayNode)currentNode;
            ArrayList<Integer> indexesToRemove = new ArrayList<Integer>();
            for (int i = 0; i < currentArrayNode.size(); ++i) {
                Map.Entry field;
                JsonNode itemNode = currentArrayNode.get(i);
                if (!itemNode.isObject() || itemNode.size() != 1 || !((String)(field = (Map.Entry)itemNode.fields().next()).getKey()).equals("...") || !((JsonNode)field.getValue()).isTextual()) continue;
                String digest = ((JsonNode)field.getValue()).asText();
                this.markDigestAsVisited(digest, visitedDigests);
                String disclosure = this.disclosures.get(digest);
                if (disclosure != null) {
                    visitedDisclosureStrings.add(disclosure);
                    JsonNode claimValue = this.validateArrayElementDigestDisclosureFormat(disclosure);
                    currentArrayNode.set(i, claimValue);
                    continue;
                }
                indexesToRemove.add(i);
            }
            indexesToRemove.forEach(arg_0 -> ((ArrayNode)currentArrayNode).remove(arg_0));
        }
        for (JsonNode childNode : currentNode) {
            this.validateViaRecursiveDisclosing(childNode, visitedDigests, visitedDisclosureStrings);
        }
        return currentNode;
    }

    private void markDigestAsVisited(String digest, Set<String> visitedDigests) throws SdJwtVerificationException {
        if (!visitedDigests.add(digest)) {
            throw new SdJwtVerificationException("A digest was encounted more than once: " + digest);
        }
    }

    private VisibleSdJwtClaim validateSdArrayDigestDisclosureFormat(String disclosure) throws SdJwtVerificationException {
        String claimName;
        ArrayNode arrayNode = SdJwtUtils.decodeDisclosureString(disclosure);
        if (arrayNode.size() != 3) {
            throw new SdJwtVerificationException("A field disclosure must contain exactly three elements");
        }
        List<String> denylist = List.of("_sd", "...");
        if (denylist.contains(claimName = arrayNode.get(1).asText())) {
            throw new SdJwtVerificationException("Disclosure claim name must not be '_sd' or '...'");
        }
        return VisibleSdJwtClaim.builder().withClaimName(arrayNode.get(1).asText()).withClaimValue(arrayNode.get(2)).build();
    }

    private JsonNode validateArrayElementDigestDisclosureFormat(String disclosure) throws SdJwtVerificationException {
        ArrayNode arrayNode = SdJwtUtils.decodeDisclosureString(disclosure);
        if (arrayNode.size() != 2) {
            throw new SdJwtVerificationException("An array element disclosure must contain exactly two elements");
        }
        return arrayNode.get(1);
    }

    private void validateDisclosuresVisits(Set<String> visitedDisclosureStrings) throws SdJwtVerificationException {
        if (visitedDisclosureStrings.size() < this.disclosures.size()) {
            throw new SdJwtVerificationException("At least one disclosure is not protected by digest");
        }
    }

    private void preventKeyBindingJwtReplay(KeyBindingJwtVerificationOpts keyBindingJwtVerificationOpts) throws SdJwtVerificationException {
        JsonNode nonce = this.keyBindingJwt.getPayload().get("nonce");
        if (nonce == null || !nonce.isTextual() || !nonce.asText().equals(keyBindingJwtVerificationOpts.getNonce())) {
            throw new SdJwtVerificationException("Key binding JWT: Unexpected `nonce` value");
        }
        JsonNode aud = this.keyBindingJwt.getPayload().get("aud");
        if (aud == null || !aud.isTextual() || !aud.asText().equals(keyBindingJwtVerificationOpts.getAud())) {
            throw new SdJwtVerificationException("Key binding JWT: Unexpected `aud` value");
        }
    }

    private void validateKeyBindingJwtSdHashIntegrity() throws SdJwtVerificationException {
        Objects.requireNonNull(this.sdJwtVpString);
        JsonNode sdHash = this.keyBindingJwt.getPayload().get("sd_hash");
        if (sdHash == null || !sdHash.isTextual()) {
            throw new SdJwtVerificationException("Key binding JWT: Claim `sd_hash` missing or not a string");
        }
        int lastDelimiterIndex = this.sdJwtVpString.lastIndexOf("~");
        String toHash = this.sdJwtVpString.substring(0, lastDelimiterIndex + 1);
        String digest = SdJwtUtils.hashAndBase64EncodeNoPad(toHash.getBytes(), this.issuerSignedJwt.getSdHashAlg());
        if (!digest.equals(sdHash.asText())) {
            throw new SdJwtVerificationException("Key binding JWT: Invalid `sd_hash` digest");
        }
    }
}

