/*
 * Copyright 2018-2018 adorsys GmbH & Co KG
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.adorsys.psd2.xs2a.core.consent;

import lombok.Value;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.Objects;

/**
 * This class is used as a container of some binary data to be used on SPI level.
 * SPI developers may save here necessary information, that will be stored and encrypted in consent.
 * SPI developer shall not use this class without consentId!
 */
@Value
public class AspspConsentData {
    /**
     * Encrypted data that may be stored in the consent management system in the consent linked to a request.<br>
     * May be null if consent does not contain such data, or request isn't done from a workflow with a consent
     */
    @Nullable
    private final byte[] aspspConsentDataBytes;

    @NotNull
    private final String consentId;

    /**
     * Checks whether consent data is contains information or not
     *
     * @return <code>true</code> if consent data is empty. <code>false</code> otherwise.
     */
    public boolean isEmptyConsentData() {
        return aspspConsentDataBytes == null && StringUtils.isBlank(consentId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        AspspConsentData that = (AspspConsentData) o;

        return Arrays.equals(aspspConsentDataBytes, that.aspspConsentDataBytes) && Objects.equals(consentId, that.getConsentId());
    }

    @Override
    public int hashCode() {
        return Objects.hash(aspspConsentDataBytes, consentId);
    }
}

