/*
 * Copyright 2018-2018 adorsys GmbH & Co KG
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package de.adorsys.aspsp.aspspmockserver.service;

import de.adorsys.aspsp.aspspmockserver.repository.PsuRepository;
import de.adorsys.psd2.aspsp.mock.api.account.AspspAccountBalance;
import de.adorsys.psd2.aspsp.mock.api.account.AspspAccountDetails;
import de.adorsys.psd2.aspsp.mock.api.psu.Psu;
import lombok.AllArgsConstructor;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.Currency;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@Service
@AllArgsConstructor
public class AccountService {
    private final PsuRepository psuRepository;

    /**
     * Adds new account to corresponding PSU by it's primary ASPSP identifier
     *
     * @param psuId          PSU's primary ASPSP identifier
     * @param accountDetails account details to be added
     * @return Optional of saved account details
     */
    public Optional<AspspAccountDetails> addAccount(String psuId, AspspAccountDetails accountDetails) {
        return Optional.ofNullable(psuRepository.findOne(psuId))
                   .map(psu -> addAccountToPsuAndSave(psu, accountDetails))
                   .flatMap(psu -> findAccountInPsuById(psu, accountDetails.getResourceId()));
    }

    /**
     * Returns a list of all accounts of all PSUs at this ASPSP (DEBUG ONLY!)
     *
     * @return list of all accounts at ASPSP
     */
    public List<AspspAccountDetails> getAllAccounts() {
        return psuRepository.findAll().stream()
                   .flatMap(psu -> psu.getAccountDetailsList().stream())
                   .collect(Collectors.toList());
    }

    /**
     * Returns account details by accounts primary ASPSP identifier
     *
     * @param accountId accounts primary ASPSP identifier
     * @return Optional of account details
     */
    public Optional<AspspAccountDetails> getAccountById(String accountId) {
        return psuRepository.findPsuByAccountDetailsList_ResourceId(accountId)
                   .flatMap(psu -> findAccountInPsuById(psu, accountId));
    }

    /**
     * Returns a list of account details containing requested IBAN
     *
     * @param iban account IBAN
     * @return list of account details
     */
    public List<AspspAccountDetails> getAccountsByIban(String iban) {
        return psuRepository.findPsuByAccountDetailsList_Iban(iban)
                   .map(Psu::getAccountDetailsList)
                   .orElseGet(Collections::emptyList);
    }

    /**
     * Returns an aspsp account Id by given IBAN And Currency
     *
     * @param iban     account IBAN
     * @param currency currency
     * @return aspsp account id
     */
    Optional<String> getAccountIdByIbanAndCurrency(String iban, Currency currency) {
        return psuRepository.findPsuByAccountDetailsList_Iban(iban)
                   .flatMap(psu -> psu.getAccountDetailsList().stream()
                                       .filter(aD -> aD.getCurrency() == currency)
                                       .findFirst()
                                       .map(AspspAccountDetails::getAspspAccountId)
                   );
    }

    Optional<String> getPsuIdByIban(String iban) {
        return psuRepository.findPsuByAccountDetailsList_Iban(iban)
                   .map(Psu::getPsuId);
    }

    Optional<Psu> getPsuByPsuId(String psuId) {
        return psuRepository.findByPsuId(psuId);
    }

    /**
     * Returns a list of balances for account represented by its primary ASPSP identifier
     *
     * @param accountId accounts primary ASPSP identifier
     * @return list of account balances
     */
    public List<AspspAccountBalance> getAccountBalancesById(String accountId) {
        return psuRepository.findPsuByAccountDetailsList_ResourceId(accountId)
                   .flatMap(psu -> findAccountInPsuById(psu, accountId))
                   .map(AspspAccountDetails::getBalances)
                   .orElseGet(Collections::emptyList);
    }

    /**
     * Returns a list of PSU's account details by PSU's primary ASPSP identifier
     *
     * @param psuId PSU's primary ASPSP identifier
     * @return list of account details
     */
    public List<AspspAccountDetails> getAccountsByPsuId(String psuId) {
        return psuRepository.findByPsuId(psuId)
                   .map(Psu::getAccountDetailsList)
                   .orElseGet(Collections::emptyList);
    }

    /**
     * Deletes account by its primary ASPSP identifier
     *
     * @param accountId accounts primary ASPSP identifier
     */
    public void deleteAccountById(String accountId) {
        psuRepository.findPsuByAccountDetailsList_ResourceId(accountId)
            .map(psu -> getPsuWithFilteredAccountListById(psu, accountId))
            .map(psuRepository::save);
    }

    Optional<AspspAccountDetails> updateAccount(AspspAccountDetails accountDetails) {
        return Optional.ofNullable(accountDetails.getResourceId())
                   .flatMap(psuRepository::findPsuByAccountDetailsList_ResourceId)
                   .map(psu -> updateAccountInPsu(psu, accountDetails))
                   .flatMap(psu -> findAccountInPsuById(psu, accountDetails.getResourceId()));
    }

    private Psu updateAccountInPsu(Psu psu, AspspAccountDetails accountDetails) {
        Psu filteredPsu = getPsuWithFilteredAccountListById(psu, accountDetails.getResourceId());
        return addAccountToPsuAndSave(filteredPsu, accountDetails);
    }

    private Optional<AspspAccountDetails> findAccountInPsuById(Psu psu, String accountId) {
        return psu.getAccountDetailsList().stream()
                   .filter(acc -> acc.getResourceId().equals(accountId))
                   .findFirst();
    }

    private Psu getPsuWithFilteredAccountListById(Psu psu, String accountId) {
        psu.setAccountDetailsList(getFilteredAccountDetailsListFromPsuById(psu, accountId));
        return psu;
    }

    private Psu addAccountToPsuAndSave(Psu psu, AspspAccountDetails accountDetails) {
        psu.getAccountDetailsList().add(accountDetails);
        return psuRepository.save(psu);
    }

    private List<AspspAccountDetails> getFilteredAccountDetailsListFromPsuById(Psu psu, String accountId) {
        return psu.getAccountDetailsList().stream()
                   .filter(ad -> !ad.getResourceId().equals(accountId))
                   .collect(Collectors.toList());
    }
}
