package de.adorsys.multibanking.web.user;

import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RestController;

import de.adorsys.multibanking.domain.UserAgentCredentials;
import de.adorsys.multibanking.exception.InvalidBankAccessException;
import de.adorsys.multibanking.exception.ResourceNotFoundException;
import de.adorsys.multibanking.exception.domain.ErrorConstants;
import de.adorsys.multibanking.exception.domain.MultibankingError;
import de.adorsys.multibanking.service.UserAgentCredentialsService;
import de.adorsys.multibanking.web.annotation.UserResource;
import de.adorsys.multibanking.web.common.BaseController;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;

/**
 * @author fpo 2018-04-04 07:01
 */
@UserResource
@RestController
@RequestMapping(path = UserAgentCredentialsController.BASE_PATH)
@Api(value = UserAgentCredentialsController.BASE_PATH, 
	tags = "MB-002 - User Agent Credentials", description="User agent credentials are used to encryp user data stored locally on a user device. User agent credentials can only be stored or loaded when the user is logged into the application.")
public class UserAgentCredentialsController extends BaseController {
	public static final String BASE_PATH = BaseController.BASE_PATH + "/useragents/{userAgentId}"; 
	
    private final static Logger LOGGER = LoggerFactory.getLogger(UserAgentCredentialsController.class);
    
    @Autowired
    private UserAgentCredentialsService userAgentCredentialsService;

    @RequestMapping(method = RequestMethod.GET, produces =MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ApiOperation(value="Load user agent credentials", notes = "Loads the user credentials object with the given userAgentId.")
    @ApiResponses(value = { @ApiResponse(code = HttpServletResponse.SC_OK, 
    		message = "Ok", response=UserAgentCredentials.class),
            @ApiResponse(code = HttpServletResponse.SC_UNAUTHORIZED, message = ErrorConstants.ERR_HTTP_CODE_UNAUTHENTICATED_DOC, response = MultibankingError.class),
    		@ApiResponse(code = HttpServletResponse.SC_NOT_FOUND, message = ResourceNotFoundException.RENDERED_MESSAGE_KEY, response = MultibankingError.class),
    		@ApiResponse(code = HttpServletResponse.SC_FORBIDDEN, message = InvalidBankAccessException.MESSAGE_DOC, response = MultibankingError.class),
    		@ApiResponse(code = HttpServletResponse.SC_BAD_REQUEST, message = ErrorConstants.ERR_HTTP_CODE_BAD_REQUEST_DOC, response = MultibankingError.class)})
    public ResponseEntity<UserAgentCredentials> getUserAgentCredentials(
    		@ApiParam(name = "userAgentId",  required=true,
    	    	value = "The unique identifier of the requesting user agent, in the scope of this user. Must match the one given while storing the user agent credentials object.", 
    	    	example="3c149076-13c4-4190-ace3-e30bf8f61526")
    		@PathVariable String userAgentId) {
    	UserAgentCredentials credentials = userAgentCredentialsService.load(userAgentId);
        return returnDocument(credentials, MediaType.APPLICATION_JSON_UTF8);
    }
    
    @RequestMapping(method = RequestMethod.PUT, consumes =MediaType.APPLICATION_JSON_UTF8_VALUE)
    @ApiOperation(value="Update user agent credentials", notes = "Create or updates the user credentials object associated with the given userAgentId.")
    @ApiResponses(value = { @ApiResponse(code = HttpServletResponse.SC_NO_CONTENT, 
		message = "User agent credential updated."),
    @ApiResponse(code = HttpServletResponse.SC_UNAUTHORIZED, message = ErrorConstants.ERR_HTTP_CODE_UNAUTHENTICATED_DOC, response = MultibankingError.class),
	@ApiResponse(code = HttpServletResponse.SC_FORBIDDEN, message = InvalidBankAccessException.MESSAGE_DOC, response = MultibankingError.class),
	@ApiResponse(code = HttpServletResponse.SC_BAD_REQUEST, message = ErrorConstants.ERR_HTTP_CODE_BAD_REQUEST_DOC, response = MultibankingError.class)})
    public HttpEntity<Void> postUserAgentCredentials(
    		@ApiParam(name = "userAgentId",  required=true,
        	value = "The unique identifier of the requesting user agent, in the scope of this user. Must be generated by the client.", 
        	example="3c149076-13c4-4190-ace3-e30bf8f61526")
    		@PathVariable String userAgentId, 
    		@ApiParam(name = "userAgentCredentials",  required=true,
        	value = "The object containing the user agent crendentials for the given user agent.")
    		@RequestBody UserAgentCredentials userAgentCredentials) {
		userAgentCredentials.setUserAgentId(userAgentId);
		userAgentCredentialsService.store(userAgentCredentials);
		LOGGER.debug("User agent credential stored for " + userId());
		return new ResponseEntity<>(HttpStatus.NO_CONTENT);
    }
}
