package de.fiveminds.client.utility;

import java.net.http.HttpResponse.BodyHandler;
import java.net.http.HttpResponse.BodySubscriber;
import java.net.http.HttpResponse.BodySubscribers;
import java.net.http.HttpResponse.ResponseInfo;
import java.nio.charset.Charset;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import de.fiveminds.client.dataModels.processInstance.BpmnError;
import de.fiveminds.client.errors.BaseError;
import de.fiveminds.client.errors.ErrorCodes;
import de.fiveminds.client.errors.ServerErrors.InternalServerError;
import lombok.Data;
import lombok.RequiredArgsConstructor;;

@RequiredArgsConstructor
public class DynamicBodyHandler<TResult> implements BodyHandler<TResult> {
	private static final Logger logger = Logger.getLogger("DynamicBodyHandler");
	private static final int httpSuccessResponseCode = 200;
	private static final int httpRedirectResponseCode = 300;
	private static final ObjectMapper objectMapper = new ObjectMapper();
	private final Class<TResult> resultClass;

	@Override
	public BodySubscriber<TResult> apply(ResponseInfo responseInfo) {
		return BodySubscribers.mapping(
				BodySubscribers.ofString(Charset.defaultCharset()), 
				jsonString -> {
					logger.log(Level.FINE, "Receiving json: ", jsonString);
					
					if (this.responseIsAnError(responseInfo)) {
						this.createAndThrowError(responseInfo.statusCode(), jsonString);
					}
					
					if (resultClass == Void.class) {
						return null;
					}
					
					try {
						return objectMapper.readValue(jsonString, resultClass);
					} catch (JsonProcessingException error) {
						throw new RuntimeException("Failed to deserialize string to class " + resultClass.getName() + ".", error);
					}
				});
	}

	private boolean responseIsAnError(ResponseInfo responseInfo) {
		return responseInfo.statusCode() < httpSuccessResponseCode || responseInfo.statusCode() >= httpRedirectResponseCode;
	}
	
	private void createAndThrowError(int statusCode, String body) {
		BaseError deserializedError = BaseError.deserialize(body);
		if (BaseError.isEngineError(deserializedError)) {
			throw deserializedError;
		}
		
		if (deserializedError.additionalInformation != null) {
			ObjectMapper objMapper = new ObjectMapper();
			BpmnErrorInfo bpmnErrorInfo;
			bpmnErrorInfo = objMapper.convertValue(deserializedError.additionalInformation, BpmnErrorInfo.class);
			
			BpmnError newError = new BpmnError(bpmnErrorInfo.err.name, bpmnErrorInfo.err.code, bpmnErrorInfo.err.message, bpmnErrorInfo.err.stackTrace);
			newError.additionalInformation = bpmnErrorInfo.err.additionalInformation;
			
			throw newError;
		}
		
		InternalServerError customError = new InternalServerError(deserializedError.getMessage() == null ? "Failed to execute request" : deserializedError.getMessage(), null);
		customError.additionalInformation = deserializedError.additionalInformation;
		customError.code = ErrorCodes.fromValue(statusCode);
		customError.setStackTrace(deserializedError.getStackTrace());
		
		throw customError;
	}
	
	@Data
	@JsonIgnoreProperties(ignoreUnknown = true)
	public static class BpmnErrorInfo {
		public BpmnErrorTemplate err;
		
		@Data
		@JsonIgnoreProperties(ignoreUnknown = true)
		public static class BpmnErrorTemplate {
			public String name;
			public String code;
			public String message;
			public String stackTrace;
			public String additionalInformation;
		}
	}
}
