package de.fiveminds.client.lib;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import de.fiveminds.client.dataModels.iam.Identity;
import de.fiveminds.client.engineEvents.Subscription;
import de.fiveminds.client.errors.UnauthorizedError;
import de.fiveminds.client.types.SocketSettings;
import de.fiveminds.client.utility.UriUtils;

import java.util.UUID;

import io.socket.client.IO.Options;
import io.socket.client.Manager;
import io.socket.client.Socket;
import io.socket.client.SocketOptionBuilder;
import io.socket.emitter.Emitter.Listener;
import io.socket.engineio.client.Transport;
import lombok.NonNull;

public class SocketIoManager implements AutoCloseable {	
	private final URI engineUrl;
	private final Manager manager;
	
	private HashMap<String, Socket> socketCollection = new HashMap<String, Socket>();
	private HashMap<String, Listener> subscriptionCollection = new HashMap<String, Listener>();
	
	public SocketIoManager(URI engineUrl) throws URISyntaxException {
		this.engineUrl = engineUrl.getScheme() == null
				? UriUtils.replaceInUri(engineUrl, "http", null, null, null, null)
				: engineUrl;
		
		URI socketUri = UriUtils.replaceInUri(engineUrl, null, null, "/" + SocketSettings.namespace, null, null);
		this.manager = new Manager(socketUri).open();
	}
	
	public void disconnectSocket(Identity identity) {
		Socket socketForIdentity = this.getSocketForIdentity(identity);
		if (socketForIdentity == null) {
			return;
		}
		
		socketForIdentity.disconnect().close();
		socketCollection.remove(identity.getUserId());
	}
	
	public Subscription createSocketIoSubscription(Identity identity, String eventName, Listener callback, boolean subscribeOnce) throws UnauthorizedError, URISyntaxException {
		Socket socketForIdentity = this.createSocketForIdentity(identity);
		
		if (subscribeOnce) {
			socketForIdentity.once(eventName, callback);
		} else {
			socketForIdentity.on(eventName, callback);
		}
		
		String subscriptionId = UUID.randomUUID().toString();
		Subscription sub = new Subscription(subscriptionId, eventName, subscribeOnce);
		
		this.subscriptionCollection.put(subscriptionId, callback);
		
		return sub;
	}
	
	public void removeSocketIoSubscription(Identity identity, Subscription subscription) {
		Socket socketForIdentity = this.getSocketForIdentity(identity);
		if (socketForIdentity == null) {
			return;
		}
		
		Listener callbackToRemove = this.subscriptionCollection.getOrDefault(subscription.getId(), null);
		if (callbackToRemove == null) {
			return;
		}
		
		socketForIdentity.off(subscription.getEventName(), callbackToRemove);
		
		this.socketCollection.remove(subscription.getId());
	}
	
	public Socket getSocketForIdentity(@NonNull Identity identity) {
		return this.socketCollection.getOrDefault(identity.getUserId(), null);
	}
	
	private Socket createSocketForIdentity(@NonNull Identity identity) throws UnauthorizedError, URISyntaxException {
		Socket existingSocket = this.getSocketForIdentity(identity);
		if (existingSocket != null) {
			return existingSocket;
		}
		
		boolean noAuthTokenProvided = identity.getToken() == null || "".equals(identity.getToken());
		if (noAuthTokenProvided) {
			throw new UnauthorizedError(identity);
		}
		
		String urlPath = engineUrl.getPath();				
		Map<String, List<String>> extraHeaders = new HashMap<String, List<String>>();
		extraHeaders.put("Authorization", List.of(identity.getToken()));
		extraHeaders.put("userId", List.of(identity.getUserId()));
		
		Options options = SocketOptionBuilder.builder()
				.setPath(urlPath == null ? "/socket.io" : urlPath + "/socket.io")
				.build();
		
		Map<String, Transport.Options> transportOptions = new HashMap<String, Transport.Options>();
		transportOptions.put("polling", SocketOptionBuilder.builder()
				.setExtraHeaders(extraHeaders)
				.build());
		
		options.transportOptions = transportOptions;
		Socket sock = manager.socket(SocketSettings.namespace, options).connect();
		
		this.socketCollection.put(identity.getUserId(), sock);
		return sock;
	}

	@Override
	public void close() {
		
		Iterator<Entry<String, Socket>> iterator = socketCollection.entrySet().iterator();
		while(iterator.hasNext()) {
			Entry<String, Socket> socketEntry = iterator.next();
			try {
				socketEntry.getValue().disconnect().close();
			} catch (Exception e) {
				// TODO: properly log this
				e.printStackTrace();
			}
			iterator.remove();
		}
		
		this.manager.off();
	}
}
