package de.fiveminds.client.errors;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Builder.Default;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;

@Data
@AllArgsConstructor
@NoArgsConstructor
@Builder
public class BaseError extends RuntimeException {
	@NonNull public ErrorCodes code;
	public Object additionalInformation;
	public ErrorCategory category;
	@Default public boolean fatal = false;
	String callStack;
	
	@Default public boolean isEngineError = true;
	
	public BaseError(
			@NonNull ErrorCodes code,
			@NonNull String message,
			ErrorCategory category) {
		super(setCategoryPrefix(message, category));
		this.code = code;
		this.category = category;
	}
	
	@JsonCreator public static BaseError deserialize(String value) {
		Map<String, Object> errorInfo;
		
		try {
			errorInfo = new ObjectMapper().readValue(value, ErrorInfo.class);
		} catch (Exception error) {
			throw new RuntimeException("Error while deserializing error: Couldn't parse string: " + error.getMessage(), error);
		}
		
		final boolean errorClassUndefined = !errorInfo.containsKey("errorClassName");
		final boolean codeUndefined = !errorInfo.containsKey("code");
		final boolean callStackUndefined = !errorInfo.containsKey("callStack");
		final boolean errorTypeUndefined = !errorInfo.containsKey("errorType");
		
		final boolean structureIsIncorrect = errorClassUndefined || codeUndefined || callStackUndefined || errorTypeUndefined;
		
		if (structureIsIncorrect) {
			throw new RuntimeException("Error while deserializing error: Serialized object has an incompatible structure.");
		}
		
		BaseError errorClass;
		try {
			errorClass = instanciateError((String) errorInfo.get("errorClassName"), (String) errorInfo.get("message"));
		} catch (NoSuchMethodException | SecurityException | InstantiationException | IllegalAccessException
				| IllegalArgumentException | InvocationTargetException error) {
			throw new RuntimeException("Error while deserializing error: Couldn't create instance of class: " + errorInfo.get("errorClassName"), error);
		}
		errorClass.setCallStack((String) errorInfo.get("callStack"));
		errorClass.setCode(ErrorCodes.fromValue((Integer) errorInfo.getOrDefault("code", null)));
		errorClass.isEngineError = (boolean)errorInfo.getOrDefault("isEngineError", false);
		errorClass.setAdditionalInformation(errorInfo.getOrDefault("additionalInformation", null));
		errorClass.setFatal((boolean)errorInfo.getOrDefault("fatal", false));

		return errorClass;
	}
	
	@JsonValue public String serialize() throws IllegalArgumentException, IllegalAccessException, JsonProcessingException {
		Map<String, Object> map = new HashMap<String, Object>();
		
		map.put("errorClassName", this.getClass().getSimpleName());
		map.put("code", this.code);
		map.put("message", this.getMessage());
		map.put("callStack", this.getStackTrace());
		map.put("additionalInformation", this.additionalInformation);
		map.put("errorType", "Error");
		for (Field field : this.getClass().getDeclaredFields()) {
			field.setAccessible(true);
			map.putIfAbsent(field.getName(), field.get(this));
		}
		
		return new ObjectMapper().writeValueAsString(map);
	}
	
	private static BaseError instanciateError(@NonNull String errorClassName, @NonNull String message) throws NoSuchMethodException, SecurityException, InstantiationException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		Map<String, Class<?>> errorClasses = new HashMap<String, Class<?>>();
		for (Class<?> errorClass : ClientErrors.class.getClasses()) {
			errorClasses.put(errorClass.getSimpleName(), errorClass);
		}
		for (Class<?> errorClass : InformationErrors.class.getClasses()) {
			errorClasses.put(errorClass.getSimpleName(), errorClass);
		}
		for (Class<?> errorClass : RedirectErrors.class.getClasses()) {
			errorClasses.put(errorClass.getSimpleName(), errorClass);
		}
		for (Class<?> errorClass : ServerErrors.class.getClasses()) {
			errorClasses.put(errorClass.getSimpleName(), errorClass);
		}
		
		if (!errorClasses.containsKey(errorClassName)) {
			throw new RuntimeException("Error while deserializing error: " + errorClassName + " is not a known error type.");
		}
		
		Class<?> errorClass = errorClasses.get(errorClassName);
		Constructor<?> errorConstructor = errorClass.getConstructor(String.class, ErrorCategory.class);
		Object error = errorConstructor.newInstance(message, null);
		return (BaseError) error;
	}
	
	private static class ErrorInfo extends HashMap<String, Object> {

		private static final long serialVersionUID = -451083304472905964L;
	}
	
	private static String setCategoryPrefix(@NonNull String message, ErrorCategory category) {
		boolean messageIncludesCategoryAlready = category == null || message.startsWith("Failure in " + category);
		
		if (messageIncludesCategoryAlready) {
			return message;
		}
		
		return "Failure in " + category + ": " + message;
	}
	
	public static boolean isEngineError(Exception error) {
		if (error instanceof BaseError) {
			return ((BaseError) error).isEngineError();
		}
		
		return false;
	}
	
	public static enum ErrorCategory {
		process,
		setup,
		extension,
	}
}
