package de.fiveminds.client.clients;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TimeZone;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import de.fiveminds.client.InternalHttpClient;
import de.fiveminds.client.dataModels.iam.Identity;
import de.fiveminds.client.engineEvents.Subscription;
import de.fiveminds.client.errors.UnauthorizedError;
import de.fiveminds.client.lib.SocketIoManager;
import de.fiveminds.client.types.RestSettings;
import de.fiveminds.client.utility.UriUtils;
import io.socket.client.Socket;
import io.socket.emitter.Emitter.Listener;
import lombok.NonNull;

public class BaseClient implements IBaseClient {
	private final URI baseUrl;
	private final URI engineUrl;
	public final Identity identity;
	
	protected InternalHttpClient httpClient;
	protected SocketIoManager socketIoManager;
	
	protected BaseClient(
			@NonNull URI engineUrl, 
			@NonNull Identity identity, 
			SocketIoManager socketIoManager) 
					throws UnknownHostException, URISyntaxException {
		this.engineUrl = engineUrl;
		this.identity = identity;
		
		this.socketIoManager = socketIoManager;
		this.baseUrl = UriUtils.extendPath(engineUrl, RestSettings.baseRoute + "/v1");
		this.httpClient = new InternalHttpClient(engineUrl);
	}

	@Override
	public boolean connected() {
		if (this.socketIoManager == null) {
			return true;
		}
		
		Socket socketIoClient = this.socketIoManager.getSocketForIdentity(identity);
		return socketIoClient == null ? false : socketIoClient.connected();
	}

	@Override
	public URI getEngineUrl() {
		return engineUrl;
	}

	@Override
	public boolean isSocketConnected(Identity identity) {
		Identity identityToUse = identity == null ? this.identity : identity;
		Socket socketIoClient = this.socketIoManager.getSocketForIdentity(identityToUse);
		return socketIoClient == null ? false : socketIoClient.connected();
	}

	@Override
	public void disconnectSocket(@NonNull Identity identity) {
		if (this.socketIoManager == null) {
			return;
		}
		
		this.socketIoManager.disconnectSocket(identity);
	}

	@Override
	public void close() throws Exception {
		if (this.socketIoManager == null) {
			return;
		}
		
		this.socketIoManager.close();
	}
	
	protected <TQuery, TSort> @NonNull URI buildUrl(@NonNull String url, int offset, int limit, TQuery query, TSort sortSettings) {
		URI finalUrl = this.baseUrl;
		
		try {
			if (finalUrl.getScheme() == null) {
				finalUrl = UriUtils.replaceInUri(finalUrl, "http", null, null, null, null);
			}
	
			finalUrl = UriUtils.extendPath(finalUrl, url);
			
			if (query != null) {
				finalUrl = this.encodeQueryAndAddParametersToUrl(finalUrl, query);
			}
			
			if (sortSettings != null) {
				String jsonString = new ObjectMapper().writeValueAsString(sortSettings);
				String encodedSortSettings = this.encodeURIComponent(jsonString);
				finalUrl = this.addParameterToUrl(finalUrl, "sortSettings", encodedSortSettings);
			}
			
			if (offset > 0) {
				finalUrl = this.addParameterToUrl(finalUrl, "offset", offset);
			}
			
			if (limit > 0) {
				finalUrl = this.addParameterToUrl(finalUrl, "limit", limit);
			}
		} catch(Exception e) {
			throw new RuntimeException("Could not build url: ", e);
		}

		return finalUrl;
	}
	
	protected List<String> createRequestAuthHeaders(Identity identity) {
		Identity identityToUse = identity == null ? this.identity : identity;
		
		List<String> requestAuthHeaders = new LinkedList<String>();
		requestAuthHeaders.add("Authorization");
		requestAuthHeaders.add("Bearer " + identityToUse.getToken());
		
		if (identityToUse.getAnonymousSessionId() != null) {
			requestAuthHeaders.add("anonymous-session-id");
			requestAuthHeaders.add(identityToUse.getAnonymousSessionId());
		}
		
		if (identityToUse.getUserId() != null) {
			requestAuthHeaders.add("user-id");
			requestAuthHeaders.add(identityToUse.getUserId());
		}
		
		return requestAuthHeaders;
	}
	
	protected @NonNull Subscription createSocketIoSubscription(@NonNull String route, @NonNull Listener callback, Boolean subscribeOnce, Identity identity) throws UnauthorizedError, URISyntaxException {
		Identity identityToUse = identity == null ? this.identity : identity;
		return this.socketIoManager.createSocketIoSubscription(identityToUse, route, callback, subscribeOnce == null ? false : subscribeOnce);
	}
	
	protected void removeSocketIoSubscription(@NonNull Subscription subscription, Identity identity) {
		Identity identityToUse = identity == null ? this.identity : identity;
		this.socketIoManager.removeSocketIoSubscription(identityToUse, subscription);
	}
	
	protected <TValue> @NonNull URI addParameterToUrl(@NonNull URI url, @NonNull String parameterName, @NonNull TValue parameterValue) {
		String appendQuery = parameterName + '=' + parameterValue;

        String newQuery = url.getQuery();
        if (newQuery == null) {
            newQuery = appendQuery;
        } else {
            newQuery += "&" + appendQuery;  
        }

        try {
			return new URI(url.getScheme(), url.getAuthority(),
					url.getPath(), newQuery, url.getFragment());
		} catch (URISyntaxException e) {
			throw new RuntimeException("An error occurred while extending URI " + url.toString() + " with parameter " + parameterName + " and value " + parameterValue + ".", e);
		}
	}
	
	private <TQuery> @NonNull URI encodeQueryAndAddParametersToUrl(@NonNull URI url, @NonNull TQuery query) throws JsonProcessingException, URISyntaxException, UnsupportedEncodingException {		
		Map<?, ?> map = new ObjectMapper().convertValue(query, Map.class);
		for (Entry<?, ?> param : map.entrySet()) {
			String paramName = param.getKey().toString();
			Object paramValue = param.getValue();
			
			if (paramValue == null) {
				continue;
			}
			
			String encodedParameter;
			if (paramValue.getClass().isArray()) {
				Object[] paramArray = (Object[]) paramValue;
				String[] encodedParams = new String[paramArray.length];
				for (int i = 0; i < paramArray.length; i++) {
					encodedParams[i] = this.encodeValue(paramArray[i]);
				}
				encodedParameter = String.join(",", encodedParams);
			} else {
				encodedParameter = this.encodeValue(paramValue);
			}
			
			url = this.addParameterToUrl(url, paramName, encodedParameter);
		}
		
		return url;
	}
	
	private <TValue> @NonNull String encodeValue(@NonNull TValue value) throws JsonProcessingException, UnsupportedEncodingException {
		if (value instanceof String || value instanceof Integer || value instanceof Boolean) {
			
			return encodeURIComponent(value);
		}
		
		if (value instanceof Date) {
			SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
			sdf.setTimeZone(TimeZone.getTimeZone("CET"));
			return encodeURIComponent(sdf.format(value));
		}
		
		String stringifiedValue = new ObjectMapper().writeValueAsString(value);
		return encodeURIComponent(stringifiedValue);
	}
	
	private <TValue> @NonNull String encodeURIComponent(@NonNull TValue value) throws UnsupportedEncodingException {
		String component = value.toString();
		String result = URLEncoder.encode(component, "UTF-8");
		result = result.replace("+", "%20");
		result = revertEncode(result, '~');
		result = revertEncode(result, '\'');
		result = revertEncode(result, '(');
		result = revertEncode(result, ')');
		result = revertEncode(result, '!');
		return result;
	}
	
	private @NonNull String revertEncode(@NonNull String value, char character) throws UnsupportedEncodingException {
		String enc = URLEncoder.encode(""+character, "UTF-8");
		String result = value.replace(enc, ""+character);
		return result;
	}
}
