package de.fiveminds.client;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;

import de.fiveminds.client.clients.ExternalTaskApiHttpClient;
import de.fiveminds.client.dataModels.externalTasks.ExternalTask;
import de.fiveminds.client.dataModels.iam.Identity;
import de.fiveminds.client.types.ExternalTaskWorkerConfig;
import de.fiveminds.client.types.HandleExternalTaskAction;
import de.fiveminds.client.types.WorkerErrorHandler;
import de.fiveminds.client.types.WorkerErrorHandler.ErrorType;
import de.fiveminds.client.utility.AbortController;
import de.fiveminds.client.utility.AbortController.AbortSignal;
import lombok.Getter;
import lombok.NonNull;
import lombok.Setter;

public class ExternalTaskWorker<TExternalTaskPayload, TResultPayload, TExternalTask extends ExternalTask<TExternalTaskPayload>> implements AutoCloseable {
	private static Logger logger = Logger.getLogger("external_task_worker");
		
	@Getter @NonNull private final String workerId;
	private final int lockDuration;
	@NonNull private final String topic;
	private final int maxTasks;
	private final int longpollingTimeout;
	@NonNull private final HandleExternalTaskAction<TExternalTaskPayload, TResultPayload> processingFunction;
	@NonNull private final Pattern payloadFilter;
	@NonNull private final Class<TExternalTask[]> externalTaskClass;
	
	@Getter @Setter private Identity identity;
	@Getter private boolean pollingActive = false;
	private ExternalTaskApiHttpClient externalTaskClient;
	private WorkerErrorHandler customErrorHandler;
	private AbortController abortController;
	private AbortSignal abortSignal;
	
	public ExternalTaskWorker(
			@NonNull URI engineUrl,
			@NonNull String workerId,
			int lockDuration,
			@NonNull String topic,
			@NonNull HandleExternalTaskAction<TExternalTaskPayload, TResultPayload> processingFunction,
			ExternalTaskWorkerConfig config,
			Class<TExternalTask[]> externalTaskClass) throws UnknownHostException, URISyntaxException {
		this.workerId = workerId;
		this.topic = topic;
		this.processingFunction = processingFunction;
	    this.identity = config == null || config.getIdentity() == null ? Client.dummyIdentity : config.getIdentity();
	    this.maxTasks = config == null || config.getMaxTasks() == null ? 10 : config.getMaxTasks();
	    this.longpollingTimeout = config == null || config.getLongpollingTimeout() == null ? 10000 : config.getLongpollingTimeout();
	    this.lockDuration = config == null || config.getLockDuration() == null ? 30000 : config.getLockDuration();
	    this.payloadFilter = config == null ? null : config.getPayloadFilter();
	    this.externalTaskClass = externalTaskClass;

	    this.externalTaskClient = new ExternalTaskApiHttpClient(engineUrl, this.identity, null);
	}
	
	public void onWorkerError(WorkerErrorHandler callback) {
		this.customErrorHandler = callback;
	}
	
	public CompletableFuture<Void> start() throws InterruptedException, ExecutionException {
		this.pollingActive = true;
		this.abortController = new AbortController();
		this.abortSignal = this.abortController.signal;
		return this.processExternalTasks();
	}
	
	public void stop() {
		this.pollingActive = false;
		
		if (this.abortController != null && !this.abortSignal.isAborted()) {
			this.abortController.abort();
		}
	}

	@Override
	public void close() throws Exception {
		this.externalTaskClient.close();
	}
	
	private CompletableFuture<Void> processExternalTasks() throws InterruptedException, ExecutionException {
		return CompletableFuture.runAsync(() -> {
			int errorCount = 0;
			final int maximumErrorTimeout = 30 * 1000;
			
			while (this.abortSignal != null && !this.abortSignal.isAborted()) {
				ExternalTask<TExternalTaskPayload>[] externalTasks;
				
				try {
					externalTasks = this.fetchAndLockExternalTasks().get();
					errorCount = 0;
				} catch (Exception e) {
					logger.log(Level.SEVERE, "Failed to fetch and lock External Tasks.", e);
					
					errorCount += 1;
					double timeoutOffset = (Math.floor(Math.random() * 10) + 1) * 30;
					double calculatedTimeout = Math.pow(2, errorCount) * 1000 + timeoutOffset;
					
					double timeoutToUse = Math.min(calculatedTimeout, maximumErrorTimeout);
					
					if (this.customErrorHandler != null) {
						this.customErrorHandler.run(ErrorType.fetchAndLock, e, null);
					}
					
					if (this.abortSignal != null && !this.abortSignal.isAborted()) {
						try {
							Thread.sleep((long) timeoutToUse);
						} catch (InterruptedException e1) {
							throw new RuntimeException(e1);
						}
					}

					externalTasks = new ExternalTask[0];
				}
				
				List<CompletableFuture<Void>> externalTaskFutures = new ArrayList<CompletableFuture<Void>>();
				for (ExternalTask<TExternalTaskPayload> externalTask : externalTasks) {
					externalTaskFutures.add(this.startExecutingExternalTask(externalTask));
				}
				
				try {
					CompletableFuture.allOf(externalTaskFutures.toArray(new CompletableFuture[externalTaskFutures.size()])).get();
				} catch (InterruptedException | ExecutionException e) {
					throw new RuntimeException(e);
				}
			}
		});
	}
	
	private CompletableFuture<ExternalTask<TExternalTaskPayload>[]> fetchAndLockExternalTasks() throws URISyntaxException, IOException, InterruptedException {
		CompletableFuture<ExternalTask<TExternalTaskPayload>[]> externalTasks = this.externalTaskClient.fetchAndLockExternalTasks(
				this.workerId,
				new String[] {this.topic}, 
				this.maxTasks,
				this.longpollingTimeout,
				this.lockDuration, 
				this.payloadFilter,
				this.identity,
				this.externalTaskClass);
		this.abortSignal.subscribe(() -> externalTasks.cancel(true));
		return externalTasks;
	}
	
	private CompletableFuture<Void> startExecutingExternalTask(ExternalTask<TExternalTaskPayload> externalTask) {
		ExternalTaskExecution<TExternalTaskPayload, TResultPayload> externalTaskExecution = new ExternalTaskExecution<TExternalTaskPayload, TResultPayload>(
				externalTask, 
				this.processingFunction,
				this.externalTaskClient,
				ExternalTaskWorkerConfig.builder()
					.identity(this.identity)
					.lockDuration(this.lockDuration)
					.longpollingTimeout(this.longpollingTimeout)
					.maxTasks(this.maxTasks)
					.payloadFilter(this.payloadFilter)
					.workerId(this.workerId)
					.build(),
				this.topic,
				this.abortSignal,
				this.customErrorHandler);
		
		return CompletableFuture.runAsync(() -> {
			try {
				externalTaskExecution.execute().get();
			} catch (Exception error) {
				if (this.customErrorHandler != null) {
					this.customErrorHandler.run(ErrorType.unprocessableExternalTask, error, externalTask);
				}
				
				logger.log(Level.WARNING, "External Task " + externalTask.getId() + " could not be processed.", new Object() {@Getter public Exception err = error;});
			} finally {
				externalTaskExecution.close();
			}
		});
	}
}
