package de.fiveminds.client;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import de.fiveminds.client.clients.ExternalTaskApiHttpClient;
import de.fiveminds.client.dataModels.externalTasks.ExternalTask;
import de.fiveminds.client.dataModels.externalTasks.ExternalTaskError;
import de.fiveminds.client.types.ExternalTaskWorkerConfig;
import de.fiveminds.client.types.HandleExternalTaskAction;
import de.fiveminds.client.types.WorkerErrorHandler;
import de.fiveminds.client.types.WorkerErrorHandler.ErrorType;
import de.fiveminds.client.utility.AbortController.AbortSignal;
import lombok.Data;
import lombok.Getter;
import lombok.NonNull;

public class ExternalTaskExecution<TExternalTaskPayload, TResultPayload> implements AutoCloseable {
	@NonNull private ExternalTask<TExternalTaskPayload> externalTask;
	@NonNull private HandleExternalTaskAction<TExternalTaskPayload, TResultPayload> processingFunction;
	@NonNull private ExternalTaskApiHttpClient externalTaskClient;
	@NonNull private ExternalTaskWorkerConfig config;
	@NonNull private String topic;
	@NonNull private AbortSignal abortSignal;
	
	private WorkerErrorHandler customErrorHandler;
	
	private ScheduledExecutorService interval;
	private Runnable abortSignalSubscription = () -> {};
	private CompletableFuture<Void> awaitAbortSignal;
	private Runnable abortSignalResolver = () -> {};
	
	private Logger logger;
	
	public ExternalTaskExecution(
			@NonNull ExternalTask<TExternalTaskPayload> externalTask,
			@NonNull HandleExternalTaskAction<TExternalTaskPayload, TResultPayload> processingFunction,
			@NonNull ExternalTaskApiHttpClient externalTaskClient,
			@NonNull ExternalTaskWorkerConfig config,
			@NonNull String topic,
			@NonNull AbortSignal abortSignal,
			WorkerErrorHandler customErrorHandler) {
		this.externalTask = externalTask;
		this.processingFunction = processingFunction;
		this.externalTaskClient = externalTaskClient;
		this.config = config;
		this.topic = topic;
		this.abortSignal = abortSignal;
		this.customErrorHandler = customErrorHandler;
		
		this.logger = Logger.getLogger("external_task_execution");
	}
	
	public CompletableFuture<Void> execute() throws URISyntaxException, IOException, InterruptedException, ExecutionException {
		return CompletableFuture.runAsync(() -> {
			if (this.abortSignal.isAborted()) {
				throw new RuntimeException(this.abortSignal.getException());
			}
			
			this.startAbortSignalSubscription();
			this.startExtendLockInterval();
			
			Object resultObject;
			try {
				resultObject = CompletableFuture.anyOf(new CompletableFuture[] {
						CompletableFuture.supplyAsync(() -> this.processingFunction.run(this.externalTask.getPayload(), this.externalTask, this.abortSignal)),
						this.awaitAbortSignal
				}).get();
			} catch (InterruptedException | ExecutionException e) {
				throw new RuntimeException(e);
			}
			
			if (this.abortSignal.isAborted()) {
				throw new RuntimeException(this.abortSignal.getException());
			}
			
			this.stopLockingInterval();
			
			if (resultObject != null) {
				try {
					this.processResult(resultObject).get();
				} catch (InterruptedException | ExecutionException e) {
					throw new RuntimeException(e);
				}
			}
	}).exceptionally((Throwable error) -> {
			if (error instanceof RuntimeException) {
				if (error.getCause() != null) {
					error = error.getCause();
				}
				
				this.handleError(ErrorType.processExternalTask, (Exception) error);
				try {
					this.handleExternalTaskExecutionError(error).get();
				} catch (InterruptedException | ExecutionException | URISyntaxException | IOException e) {
					throw new RuntimeException("An unexpected exception occurred while handling an exception.", e);
				}
			}
			return null;
		});
	}
	
	private CompletableFuture<Void> processResult(Object result) {
		try {
			if (result instanceof ExternalTaskError) {
				this.handleExternalTaskExecutionError(result).get();
				return CompletableFuture.failedFuture((ExternalTaskError)result);
			}
			
			if (result instanceof Exception) {
				this.handleExternalTaskExecutionError(result).get();
				return CompletableFuture.failedFuture((Exception)result);
			}
			
			return this.externalTaskClient.finishExternalTask(this.config.workerId, this.externalTask.getId(), result, this.config.identity);
		} catch (Exception error) {
			this.handleError(ErrorType.finishExternalTask, error);
			return CompletableFuture.failedFuture(error);
		}
	}
	
	private CompletableFuture<Void> handleExternalTaskExecutionError(Object error) throws URISyntaxException, IOException, InterruptedException {
		if (this.abortSignal.isAborted()) {
			return CompletableFuture.failedFuture(this.abortSignal.getException());
		}
		
		this.logger.log(Level.SEVERE, "Error raised for external task " + this.externalTask.getId() + " with topic " + this.topic + ": ", new LogInfo() {@Getter public Object err = error;});
		
		ExternalTaskError workerError;
		if (error instanceof ExternalTaskError) {
			workerError = (ExternalTaskError) error;
		} else if (error instanceof Exception) {
			Exception typedError = (Exception) error;
			workerError = ExternalTaskError.builder()
					.errorCode(typedError.getClass().getName())
					.errorMessage(typedError.getMessage())
					.errorDetails(typedError.getStackTrace())
					.build();
		} else {
			workerError = ExternalTaskError.builder()
					.errorCode("ExternalTaskExecutionError")
					.errorMessage("An error occurred while processing the external task.")
					.errorDetails("No error details available.")
					.build();
		}
		
		return this.externalTaskClient.handleError(this.config.workerId, this.externalTask.getId(), workerError, this.config.identity);
	}

	@Override
	public void close() {
		this.removeAbortSignalSubscription();
		this.stopLockingInterval();
	}
	
	private void stopLockingInterval() {
		if (this.interval != null) {
			this.interval.shutdown();
			this.interval = null;
		}
	}
	
	private void startAbortSignalSubscription() {
		this.awaitAbortSignal = new CompletableFuture<Void>();
		this.abortSignalResolver = () -> this.awaitAbortSignal.complete(null);
		
		this.abortSignalSubscription = () -> {
			if (this.abortSignalResolver != null) {
				this.abortSignalResolver.run();
			}
			
			this.close();
		};
		
		this.abortSignal.subscribe(this.abortSignalSubscription);
	}
	
	private void removeAbortSignalSubscription() {
		if (this.abortSignalSubscription != null) {
			this.abortSignal.unsubscribe(this.abortSignalSubscription);
		}
		
		if (this.abortSignalResolver != null) {
			this.abortSignalResolver.run();
		}
	}
	
	private void startExtendLockInterval() {
		final int lockExtensionBuffer = 5000;
		
		this.interval = Executors.newSingleThreadScheduledExecutor();
		this.interval.scheduleAtFixedRate(() -> this.extendLocks(this.externalTask), 0, this.config.lockDuration - lockExtensionBuffer, TimeUnit.MILLISECONDS);
	}
	
	private CompletableFuture<Void> extendLocks(ExternalTask<TExternalTaskPayload> externalTask) {
		try {
			return this.externalTaskClient.extendLock(this.config.workerId, this.externalTask.getId(), this.config.lockDuration, this.config.identity);
		} catch (Exception error) {
			this.handleError(ErrorType.extendLock, error);
			
			this.logger.log(Level.WARNING, "An error occurred while trying to extend the lock for ExternalTask " + this.externalTask.getId(), new LogInfo() {@Getter public Exception err = error;});

			return CompletableFuture.failedFuture(error);
		}
	}
	
	private void handleError(ErrorType errorType, Exception error) {
		if (this.customErrorHandler != null) {
			this.customErrorHandler.run(errorType, error, this.externalTask);
		}
	}
	
	@Data
	private class LogInfo {
		public String workerId = ExternalTaskExecution.this.config.workerId;
		public String externalTaskId = ExternalTaskExecution.this.externalTask.getId();
		public String topic = ExternalTaskExecution.this.topic;
	}
}
