package de.fiveminds.client.clients;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.net.http.HttpResponse;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import de.fiveminds.client.InternalHttpClient.RequestOptions;
import de.fiveminds.client.dataModels.flowNodeInstance.FlowNodeInstance.FlowNodeInstanceSortSettings;
import de.fiveminds.client.dataModels.flowNodeInstance.Queries.GenericFlowNodeInstanceQuery;
import de.fiveminds.client.dataModels.flowNodeInstance.Queries.UserTaskQuery;
import de.fiveminds.client.dataModels.flowNodeInstance.UserTaskInstance.UserTaskList;
import de.fiveminds.client.dataModels.flowNodeInstance.UserTaskInstance.UserTaskResult;
import de.fiveminds.client.dataModels.iam.Identity;
import de.fiveminds.client.extensionAdapter.IUserTaskExtensionAdapter;
import de.fiveminds.client.lib.SocketIoManager;
import de.fiveminds.client.processModel.model.constants.BpmnType;
import de.fiveminds.client.types.RestSettings;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NonNull;

public class UserTaskHttpClient extends BaseClient implements IUserTaskExtensionAdapter {

	public UserTaskHttpClient(@NonNull URI engineUrl, @NonNull Identity identity, SocketIoManager socketIoManager)
			throws UnknownHostException, URISyntaxException {
		super(engineUrl, identity, socketIoManager);
	}

	@Override
	public CompletableFuture<UserTaskList> query(@NonNull UserTaskQuery query, UserTaskOptions options) {
		FlowNodeInstanceSortSettings sortSettings = options == null ? null : options.getSortSettings();
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		GenericFlowNodeInstanceQuery enhancedQuery = new GenericFlowNodeInstanceQuery();
		enhancedQuery.setFlowNodeType(new BpmnType[] {BpmnType.userTask});
		enhancedQuery.setFlowNodeId(query.getFlowNodeId());
		enhancedQuery.setFlowNodeName(query.getFlowNodeName());
		enhancedQuery.setFlowNodeLane(query.getFlowNodeLane());
		enhancedQuery.setFlowNodeInstanceId(query.getFlowNodeInstanceId());
		enhancedQuery.setPreviousFlowNodeInstanceId(query.getPreviousFlowNodeInstanceId());
		enhancedQuery.setProcessDefinitionId(query.getProcessDefinitionId());
		enhancedQuery.setProcessModelId(query.getProcessModelId());
		enhancedQuery.setProcessInstanceId(query.getProcessInstanceId());
		enhancedQuery.setCorrelationId(query.getCorrelationId());
		enhancedQuery.setParentProcessInstanceId(query.getParentProcessInstanceId());
		enhancedQuery.setState(query.getState());
		enhancedQuery.setOwnerId(query.getOwnerId());
		enhancedQuery.setTriggeredByFlowNodeInstance(query.getTriggeredByFlowNodeInstance());
		
		URI url = this.buildUrl(RestSettings.Paths.queryFlowNodeInstances, options == null ? 0 : options.getOffset(), options == null ? 0 : options.getLimit(), enhancedQuery, sortSettings);
		
		
		CompletableFuture<HttpResponse<UserTaskList>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					UserTaskList.class);
	
		return httpResponse.thenApply((HttpResponse<UserTaskList> result) -> result.body());
	}

	@Override
	public CompletableFuture<Void> reserveUserTaskInstance(@NonNull Identity identity, @NonNull String flowNodeInstanceId,
			@NonNull String actualOwnerId) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.reserveUserTaskInstance.replace(RestSettings.Params.userTaskInstanceId, flowNodeInstanceId), 0, 0, null, null);
		
		@Data
		@AllArgsConstructor
		class Body {
			public String actualOwnerId;
		}
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				new Body(actualOwnerId),
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);
		
		return httpResponse.thenApply((HttpResponse<Void> result) -> result.body());
	}

	@Override
	public CompletableFuture<Void> cancelUserTaskInstanceReservation(@NonNull Identity identity, @NonNull String flowNodeInstanceId) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.cancelUserTaskInstanceReservation.replace(RestSettings.Params.userTaskInstanceId, flowNodeInstanceId), 0, 0, null, null);
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.delete(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);
		
		return httpResponse.thenApply((HttpResponse<Void> result) -> result.body());
	}

	@Override
	public CompletableFuture<Void> finishUserTask(@NonNull String userTaskInstanceId, @NonNull UserTaskResult userTaskResult,
			Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.finishUserTask.replace(RestSettings.Params.userTaskInstanceId, userTaskInstanceId), 0, 0, null, null);
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				userTaskResult,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);
		
		return httpResponse.thenApply((HttpResponse<Void> result) -> result.body());
	}

//	TODO: Fix SocketIoManager
//	@Override
//	public Subscription onUserTaskWaiting(@NonNull Listener callback, CallbackOptions options) throws UnauthorizedError, URISyntaxException {
//		return this.createSocketIoSubscription(SocketSettings.MessagePaths.userTaskWaiting, callback, options == null ? null : options.getSubscribeOnce(), options == null ? null : options.getIdentity());
//	}
//
//	@Override
//	public Subscription onUserTaskFinished(@NonNull Listener callback, CallbackOptions options) throws UnauthorizedError, URISyntaxException {
//		return this.createSocketIoSubscription(SocketSettings.MessagePaths.userTaskFinished, callback, options == null ? null : options.getSubscribeOnce(), options == null ? null : options.getIdentity());
//	}
//
//	@Override
//	public Subscription onUserTaskReserved(@NonNull Listener callback, CallbackOptions options) throws UnauthorizedError, URISyntaxException {
//		return this.createSocketIoSubscription(SocketSettings.MessagePaths.userTaskReserved, callback, options == null ? null : options.getSubscribeOnce(), options == null ? null : options.getIdentity());
//	}
//
//	@Override
//	public Subscription onUserTaskReservationCanceled(@NonNull Listener callback, CallbackOptions options) throws UnauthorizedError, URISyntaxException {
//		return this.createSocketIoSubscription(SocketSettings.MessagePaths.userTaskReservationCanceled, callback, options == null ? null : options.getSubscribeOnce(), options == null ? null : options.getIdentity());
//	}
//
//	@Override
//	public void removeSubscription(@NonNull Subscription subscription, Identity identity) {
//		this.removeSocketIoSubscription(subscription, identity);
//	}

}
