package utility;

import java.net.http.HttpResponse.BodyHandler;
import java.net.http.HttpResponse.BodySubscriber;
import java.net.http.HttpResponse.BodySubscribers;
import java.net.http.HttpResponse.ResponseInfo;
import java.nio.charset.Charset;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import dataModels.processInstance.BpmnError;
import errors.BaseError;
import errors.ErrorCodes;
import errors.ServerErrors.InternalServerError;
import lombok.Data;
import lombok.RequiredArgsConstructor;;

@RequiredArgsConstructor
public class DynamicBodyHandler<TResult> implements BodyHandler<TResult> {
	private static final int httpSuccessResponseCode = 200;
	private static final int httpRedirectResponseCode = 300;
	private static final ObjectMapper objectMapper = new ObjectMapper();
	private final Class<TResult> resultClass;

	@Override
	public BodySubscriber<TResult> apply(ResponseInfo responseInfo) {
		return BodySubscribers.mapping(
				BodySubscribers.ofString(Charset.defaultCharset()), 
				jsonString -> {
					if (this.responseIsAnError(responseInfo)) {
						this.createAndThrowError(responseInfo.statusCode(), jsonString);
					}
					
					if (resultClass == Void.class) {
						return null;
					}
					
					try {
						return objectMapper.readValue(jsonString, resultClass);
					} catch (JsonProcessingException error) {
						throw new RuntimeException("Failed to deserialize string to class " + resultClass.getName() + ".", error);
					}
				});
	}

	private boolean responseIsAnError(ResponseInfo responseInfo) {
		return responseInfo.statusCode() < httpSuccessResponseCode || responseInfo.statusCode() >= httpRedirectResponseCode;
	}
	
	private void createAndThrowError(int statusCode, String body) {
		BaseError deserializedError = BaseError.deserialize(body);
		if (BaseError.isEngineError(deserializedError)) {
			throw deserializedError;
		}
		
		if (deserializedError.additionalInformation != null) {
			ObjectMapper objMapper = new ObjectMapper();
			BpmnErrorInfo bpmnErrorInfo;
			try {
				bpmnErrorInfo = objMapper.readValue(deserializedError.additionalInformation, BpmnErrorInfo.class);
			} catch (JsonProcessingException error) {
				throw new RuntimeException("Could not deserialize additionalInformation.", error);
			}
			
			BpmnError newError = new BpmnError(bpmnErrorInfo.bpmnError.name, bpmnErrorInfo.bpmnError.code, bpmnErrorInfo.bpmnError.message, bpmnErrorInfo.bpmnError.stackTrace);
			newError.additionalInformation = bpmnErrorInfo.bpmnError.additionalInformation;
			
			throw newError;
		}
		
		InternalServerError customError = new InternalServerError(deserializedError.getMessage() == null ? "Failed to execute request" : deserializedError.getMessage(), null);
		customError.additionalInformation = deserializedError.additionalInformation;
		customError.code = ErrorCodes.fromValue(statusCode);
		customError.setStackTrace(deserializedError.getStackTrace());
		
		throw customError;
	}
	
	@Data
	@JsonIgnoreProperties(ignoreUnknown = true)
	public static class BpmnErrorInfo {
		public BpmnErrorTemplate bpmnError;
		
		@Data
		@JsonIgnoreProperties(ignoreUnknown = true)
		public static class BpmnErrorTemplate {
			public String name;
			public String code;
			public String message;
			public String stackTrace;
			public String additionalInformation;
		}
	}
}
