package dataModels.processInstance;

import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;

@Data
@AllArgsConstructor
@NoArgsConstructor
/**
 * Contains configurable options for starting a ProcessInstance with a client interface.
 * @param <TToken> The token to use for an initial ProcessToken.
 */
public class ProcessStartOptions<TToken> {
	/**
	 * The ID of the ProcessModel to start.
	 */
	@NonNull private String processModelId;
	
	/**
	 * The ID of a StartEvent at which the ProcessInstance should start.
     * Must be provided, if the ProcessModel has multiple StartEvents,
     * but is optional, if the ProcessModel only has one.
	 */
	private String startEventId;
	
	/**
	 * The ID of a correlation to which the ProcessInstance should belong.
	 * Default is an auto-generated uuid.
	 */
	private String correlationId;
	
	/**
	 * A set of values to use for an initial ProcessToken.
	 */
	private TToken initialToken;
	
	public static enum StartCallbackType {
		CallbackOnProcessInstanceCreated(1),
		CallbackOnProcessInstanceFinished(2),
		CallbackOnEndEventReached(3);
		
		public final int value;
		private final static Map<Integer, StartCallbackType> CONSTANTS = new HashMap<Integer, StartCallbackType>();
		
		static {
			for (StartCallbackType startCallbackType : values()) {
				CONSTANTS.put(startCallbackType.value, startCallbackType);
			}
		}
		
		private StartCallbackType(int value) {
			this.value = value;
		}
		
		@JsonValue
		public int value() {
			return this.value;
		}
		
		@JsonCreator
		public static StartCallbackType fromValue(Integer value) {
			StartCallbackType startCallbackType = CONSTANTS.get(value);
		
			if (startCallbackType == null && value != null) {
				throw new IllegalArgumentException("Cannot deserialize StartCallbackType from value " + value + ".");
			}
			
			return startCallbackType;
		}
	}
	
	@Data
	@AllArgsConstructor
	@NoArgsConstructor
	@Builder
	/**
	 * Contains configurable options for starting a ProcessInstance.
	 * @param <TToken> The token to use for an initial ProcessToken.
	 */
	public static class ProcessStartRequest <TToken> extends ProcessStartOptions<TToken> {
		/**
		 * Determines when the Engine will resolve, after a new request for starting a ProcessInstance was made.
	     *
	     * Note:
	     * If "CallbackOnEndEventReached" is selected, "endEventId" is no longer optional!
		 */
		private StartCallbackType returnOn;
		
		/**
		 * The ID of an EndEvent that should be awaited.
		 * Must be provided, of "returnOn" is set to "CallbackOnEndEventReached".
		 */
		private String endEventId;
	}
}
