package clients;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.net.http.HttpResponse;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import clients.InternalHttpClient.RequestOptions;
import dataModels.iam.Identity;
import dataModels.processDefinitions.ProcessDefinition.ProcessModel;
import dataModels.processDefinitions.ProcessDefinition.ProcessModelList;
import dataModels.processInstance.ProcessStartOptions;
import dataModels.processInstance.ProcessStartOptions.ProcessStartRequest;
import dataModels.processInstance.ProcessStartOptions.StartCallbackType;
import dataModels.processInstance.ProcessStartResponse;
import extensionAdapter.IProcessModelExtensionAdapter;
import lib.SocketIoManager;
import lombok.NonNull;
import types.RestSettings;

public class ProcessModelHttpClient extends BaseClient implements IProcessModelExtensionAdapter {

	protected ProcessModelHttpClient(@NonNull URI engineUrl, @NonNull Identity identity,
			SocketIoManager socketIoManager) throws UnknownHostException, URISyntaxException {
		super(engineUrl, identity, socketIoManager);
	}

	@Override
	public CompletableFuture<ProcessModelList> getAll(ProcessModelQueryOptions options) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.getAllProcessModels, options == null ? 0 : options.getOffset(), options == null ? 0 : options.getLimit(), null, null);

		CompletableFuture<HttpResponse<ProcessModelList>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					ProcessModelList.class);

		return httpResponse.thenApply((HttpResponse<ProcessModelList> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<ProcessModel> getById(@NonNull String processModelId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.getProcessModelById.replace(RestSettings.Params.processModelId, processModelId), 0, 0, null, null);

		CompletableFuture<HttpResponse<ProcessModel>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					ProcessModel.class);

		return httpResponse.thenApply((HttpResponse<ProcessModel> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<ProcessModel> getByProcessInstanceId(@NonNull String processInstanceId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.getProcessModelForProcessInstance.replace(RestSettings.Params.processInstanceId, processInstanceId), 0, 0, null, null);

		CompletableFuture<HttpResponse<ProcessModel>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					ProcessModel.class);

		return httpResponse.thenApply((HttpResponse<ProcessModel> httpResult) -> httpResult.body());
	}

	@Override
	public <TPayload> CompletableFuture<ProcessStartResponse> startProcessInstance(
			@NonNull ProcessStartOptions<TPayload> options, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.startProcessInstance.replace(RestSettings.Params.processModelId, options.getProcessModelId()), 0, 0, null, null);

		
		ProcessStartRequest<TPayload> body = new ProcessStartRequest<TPayload>(StartCallbackType.CallbackOnProcessInstanceCreated, null);
		body.setCorrelationId(options.getCorrelationId());
		body.setInitialToken(options.getInitialToken());
		body.setProcessModelId(options.getProcessModelId());
		body.setStartEventId(options.getStartEventId());
		
		CompletableFuture<HttpResponse<ProcessStartResponse>> httpResponse = this.httpClient.post(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				ProcessStartResponse.class);

		return httpResponse.thenApply((HttpResponse<ProcessStartResponse> httpResult) -> httpResult.body());
	}

	@Override
	public <TPayload> CompletableFuture<ProcessStartResponse> startProcessInstanceAndAwaitEndEvent(
			@NonNull ProcessStartOptions<TPayload> options, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.startProcessInstance.replace(RestSettings.Params.processModelId, options.getProcessModelId()), 0, 0, null, null);

		
		ProcessStartRequest<TPayload> body = new ProcessStartRequest<TPayload>(StartCallbackType.CallbackOnProcessInstanceFinished, null);
		body.setCorrelationId(options.getCorrelationId());
		body.setInitialToken(options.getInitialToken());
		body.setProcessModelId(options.getProcessModelId());
		body.setStartEventId(options.getStartEventId());
		
		CompletableFuture<HttpResponse<ProcessStartResponse>> httpResponse = this.httpClient.post(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				ProcessStartResponse.class);

		return httpResponse.thenApply((HttpResponse<ProcessStartResponse> httpResult) -> httpResult.body());
	}

	@Override
	public <TPayload> CompletableFuture<ProcessStartResponse> startProcessInstanceAndAwaitSpecificEndEvent(
			@NonNull ProcessStartOptions<TPayload> options, @NonNull String endEventId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.startProcessInstance.replace(RestSettings.Params.processModelId, options.getProcessModelId()), 0, 0, null, null);

		
		ProcessStartRequest<TPayload> body = new ProcessStartRequest<TPayload>(StartCallbackType.CallbackOnEndEventReached, endEventId);
		body.setCorrelationId(options.getCorrelationId());
		body.setInitialToken(options.getInitialToken());
		body.setProcessModelId(options.getProcessModelId());
		body.setStartEventId(options.getStartEventId());
		
		CompletableFuture<HttpResponse<ProcessStartResponse>> httpResponse = this.httpClient.post(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				ProcessStartResponse.class);

		return httpResponse.thenApply((HttpResponse<ProcessStartResponse> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<Void> deleteDefinitionByProcessModelId(@NonNull String processModelId,
			DeleteDefinitionByProcessModelIdOptions options) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.deleteProcessDefinitionByProcessModelId.replace(RestSettings.Params.processModelId, processModelId), 0, 0, null, null);

		
		if (options != null && options.getForce() != null) {
			url = this.addParameterToUrl(url, "force", options.getForce());
		}
		
		if (options != null && options.getDeleteAllRelatedData() != null) {
			url = this.addParameterToUrl(url, "delete_all_related_data", options.getDeleteAllRelatedData());
		}
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.delete(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<Void> enableProcessModel(@NonNull String processModelId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.enableProcessModel.replace(RestSettings.Params.processModelId, processModelId), 0, 0, null, null);

		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.post(
				url,
				null,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<Void> disableProcessModel(@NonNull String processModelId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.disableProcessModel.replace(RestSettings.Params.processModelId, processModelId), 0, 0, null, null);

		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.post(
				url,
				null,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

}
