package clients;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.net.http.HttpResponse;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import clients.InternalHttpClient.RequestOptions;
import dataModels.iam.Identity;
import dataModels.processDefinitions.ProcessDefinition;
import dataModels.processDefinitions.ProcessDefinition.ProcessModel;
import dataModels.processInstance.ProcessInstance.ProcessInstanceList;
import dataModels.processInstance.ProcessInstanceQuery;
import dataModels.processInstance.ProcessInstanceQuery.ProcessInstanceSortSettings;
import extensionAdapter.IProcessInstanceExtensionAdapter;
import lib.SocketIoManager;
import lombok.Data;
import lombok.NonNull;
import types.RestSettings;

public class ProcessInstanceHttpClient extends BaseClient implements IProcessInstanceExtensionAdapter {

	protected ProcessInstanceHttpClient(@NonNull URI engineUrl, @NonNull Identity identity,
			SocketIoManager socketIoManager) throws UnknownHostException, URISyntaxException {
		super(engineUrl, identity, socketIoManager);
	}

	@Override
	public CompletableFuture<ProcessInstanceList> query(@NonNull ProcessInstanceQuery query, ProcessInstanceQueryOptions options) {
		ProcessInstanceSortSettings sortSettings = options == null ? null : options.getSortSettings();
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.queryProcessInstances, options == null ? 0 : options.getOffset(), options == null ? 0 : options.getLimit(), query, sortSettings);

		CompletableFuture<HttpResponse<ProcessInstanceList>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					ProcessInstanceList.class);
	
		return httpResponse.thenApply((HttpResponse<ProcessInstanceList> result) -> result.body());
	}

	@Override
	public CompletableFuture<ProcessDefinition> getProcessDefinition(@NonNull String processInstanceId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(
					RestSettings.Paths.getProcessDefinitionForProcessInstance.replace(RestSettings.Params.processInstanceId, processInstanceId), 0, 0, null, null);

		CompletableFuture<HttpResponse<ProcessDefinition>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					ProcessDefinition.class);
	
		return httpResponse.thenApply((HttpResponse<ProcessDefinition> result) -> result.body());
	}

	@Override
	public CompletableFuture<ProcessModel> getProcessModel(@NonNull String processInstanceId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.getProcessModelForProcessInstance.replace(RestSettings.Params.processInstanceId, processInstanceId), 0, 0, null, null);

		CompletableFuture<HttpResponse<ProcessModel>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
					ProcessModel.class);
	
		return httpResponse.thenApply((HttpResponse<ProcessModel> result) -> result.body());
	}

	@Override
	public CompletableFuture<Void> transferOwnership(@NonNull String processInstanceId, @NonNull Identity newOwner, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.changeProcessInstanceOwner.replace(RestSettings.Params.processInstanceId, processInstanceId), 0, 0, null, null);

		HashMap<String, Object> body = new HashMap<String, Object>();
		body.put("newOwner", newOwner);
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<Void> terminateProcessInstance(@NonNull String processInstanceId, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.terminateProcessInstances.replace(RestSettings.Params.processInstanceId, processInstanceId), 0, 0, null, null);

		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				null,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<Void> retryProcessInstance(@NonNull String processInstanceId, ProcessInstanceRetryOptions options) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.retryProcessInstance.replace(RestSettings.Params.processInstanceId, processInstanceId), 0, 0, null, null);

		if (options != null && options.getFlowNodeInstanceId() != null) {
			url = this.addParameterToUrl(url, "flow_node_instance_id", options.getFlowNodeInstanceId());
		}
		
		if (options != null && options.getUpdateProcessModel() != null) {
			url = this.addParameterToUrl(url, "update_process_model", options.getUpdateProcessModel());
		}
		
		@Data
		class Body{
			public Object newStartToken = options == null ? null : options.getNewStartToken();
		}
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				new Body(),
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

	@Override
	public CompletableFuture<Void> deleteProcessInstances(@NonNull String[] processInstanceIds, boolean deleteAllRelatedData,
			Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.deleteProcessInstances, 0, 0, null, null);

		String serializedProcessInstanceIds = String.join(";", processInstanceIds);
		url = this.addParameterToUrl(url, "process_instance_ids", serializedProcessInstanceIds);
		
		if (deleteAllRelatedData) {
			url = this.addParameterToUrl(url, "delete_all_related_data", deleteAllRelatedData);
		}
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.delete(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

}
