package clients;

import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpClient.Version;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Builder.Default;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.NonNull;
import utility.DynamicBodyHandler;
import utility.DynamicBodyPublisher;

public class InternalHttpClient {
	private final @NonNull URI url;
	
	public InternalHttpClient(@NonNull URI url) {
		this.url = url;
	}
	
	public <TResponse> CompletableFuture<HttpResponse<TResponse>> get(
			@NonNull URI url,
			RequestOptions options,
			Class<TResponse> responseClass) {
		ArrayList<String> headers = this.buildHeaders(options);
		
		HttpRequest request = HttpRequest.newBuilder(url)
				.version(Version.HTTP_1_1)
				.headers(headers.toArray(new String[headers.size()]))
				.GET()
				.timeout(Duration.ofMillis(options.getTimeout()))
				.build();
		
		HttpClient httpClient = HttpClient.newBuilder().build();
		CompletableFuture<HttpResponse<TResponse>> response = httpClient.sendAsync(request, new DynamicBodyHandler<TResponse>(responseClass));
		
		return response;
	}
	
	public <TResponse> CompletableFuture<HttpResponse<TResponse>> delete(
			@NonNull URI url,
			RequestOptions options,
			Class<TResponse> responseClass) {
		ArrayList<String> headers = this.buildHeaders(options);
		
		HttpRequest request = HttpRequest.newBuilder(url)
				.version(Version.HTTP_1_1)
				.headers(headers.toArray(new String[headers.size()]))
				.DELETE()
				.timeout(Duration.ofMillis(options.getTimeout()))
				.build();
		
		HttpClient httpClient = HttpClient.newBuilder().build();
		CompletableFuture<HttpResponse<TResponse>> response = httpClient.sendAsync(request, new DynamicBodyHandler<TResponse>(responseClass));
		
		return response;
	}
	
	public <TResponse> CompletableFuture<HttpResponse<TResponse>> post(
			@NonNull URI url,
			Object payload,
			RequestOptions options,
			Class<TResponse> responseClass) {
		ArrayList<String> headers = this.buildHeaders(options);
		headers.add("Content-Type");
		headers.add("application/json");
		
		HttpRequest request = HttpRequest.newBuilder(url)
				.version(Version.HTTP_1_1)
				.headers(headers.toArray(new String[headers.size()]))
				.POST(new DynamicBodyPublisher(payload))
				.timeout(Duration.ofMillis(options.getTimeout()))
				.build();
		
		HttpClient httpClient = HttpClient.newBuilder().build();
		CompletableFuture<HttpResponse<TResponse>> response = httpClient.sendAsync(request, new DynamicBodyHandler<TResponse>(responseClass));
		
		return response;
	}
	
	public <TResponse> CompletableFuture<HttpResponse<TResponse>> put(
			@NonNull URI url,
			Object payload,
			RequestOptions options,
			Class<TResponse> responseClass) {
		ArrayList<String> headers = this.buildHeaders(options);
		headers.add("Content-Type");
		headers.add("application/json");
		
		HttpRequest request = HttpRequest.newBuilder(url)
				.version(Version.HTTP_1_1)
				.headers(headers.toArray(new String[headers.size()]))
				.PUT(new DynamicBodyPublisher(payload))
				.timeout(Duration.ofMillis(options.getTimeout()))
				.build();
		
		HttpClient httpClient = HttpClient.newBuilder().build();
		CompletableFuture<HttpResponse<TResponse>> response = httpClient.sendAsync(request, new DynamicBodyHandler<TResponse>(responseClass));
		
		return response;
	}
	
	private ArrayList<String> buildHeaders(RequestOptions options) {
		ArrayList<String> headers = new ArrayList<String>();
		headers.add("Access-Control-Allow-Origin");
		headers.add("*");
		headers.add("Content-Type");
		headers.add("application/json");
		headers.add("accept");
		headers.add("application/json");
		
		if (options == null || options.getHeaders() == null) {
			return headers;
		}
		
		headers.addAll(options.getHeaders());
		
		return headers;
	}
	
	@Data
	@AllArgsConstructor
	@NoArgsConstructor
	@Builder
	public static class RequestOptions {
		@Default public Integer timeout = 5000;
		
		@Default public Collection<String> headers = new ArrayList<String>();
	}
}
