package clients;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.net.http.HttpResponse;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;

import clients.InternalHttpClient.RequestOptions;
import dataModels.externalTasks.ExternalTask;
import dataModels.externalTasks.ExternalTaskError;
import dataModels.externalTasks.ExternalTaskRequestPayloads.ExtendLockRequestPayload;
import dataModels.externalTasks.ExternalTaskRequestPayloads.FetchAndLockRequestPayload;
import dataModels.externalTasks.ExternalTaskRequestPayloads.FinishExternalTaskRequestPayload;
import dataModels.externalTasks.ExternalTaskRequestPayloads.HandleExternalTaskErrorRequestPayload;
import dataModels.iam.Identity;
import extensionAdapter.IExternalTaskExtensionAdapter;
import lib.SocketIoManager;
import lombok.NonNull;
import types.RestSettings;

public class ExternalTaskApiHttpClient extends BaseClient implements IExternalTaskExtensionAdapter {

	public ExternalTaskApiHttpClient(@NonNull URI engineUrl, @NonNull Identity identity,
			SocketIoManager socketIoManager) throws UnknownHostException, URISyntaxException {
		super(engineUrl, identity, socketIoManager);
	}

	@Override
	public CompletableFuture<String[]> getAllDeployedTopics(Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.getAllDeployedTopics, 0, 0, null, null);
		
		CompletableFuture<HttpResponse<String[]>> httpResponse = this.httpClient.get(
				url,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				String[].class);
	
		return httpResponse.thenApply((HttpResponse<String[]> result) -> result.body());
	}

	@Override
	public <TPayload, TExternalTask extends ExternalTask<TPayload>> CompletableFuture<ExternalTask<TPayload>[]> fetchAndLockExternalTasks(@NonNull String workerId,
			@NonNull String[] topicNames, int maxTasks, int longPollingTimeout, int lockDuration, Pattern payloadFilter,
			Identity identity, @NonNull Class<TExternalTask[]> externalTaskWithPayloadClass) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.fetchAndLockExternalTasks, 0, 0, null, null);
		
		FetchAndLockRequestPayload body = FetchAndLockRequestPayload.builder()
				.workerId(workerId)
				.topicName(topicNames)
				.maxTasks(maxTasks)
				.longPollingTimeout(longPollingTimeout)
				.lockDuration(lockDuration)
				.payloadFilter(payloadFilter == null ? "/.*/" : payloadFilter.pattern())
				.build();
		
		CompletableFuture<HttpResponse<TExternalTask[]>> httpResponse = this.httpClient.post(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				externalTaskWithPayloadClass);
	
		return httpResponse.thenApply((HttpResponse<TExternalTask[]> result) -> result.body());
	}

	@Override
	public CompletableFuture<Void> extendLock(@NonNull String workerId, @NonNull String externalTaskId, int additionalDuration,
			Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.extendExternalTaskLock.replace(RestSettings.Params.externalTaskId, externalTaskId), 0, 0, null, null);
		
		ExtendLockRequestPayload body = ExtendLockRequestPayload.builder()
				.workerId(workerId)
				.additionalDuration(additionalDuration)
				.build();
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);
		
		return httpResponse.thenApply((HttpResponse<Void> result) -> result.body());
	}

	@Override
	public CompletableFuture<Void> handleError(@NonNull String workerId, @NonNull String externalTaskId, @NonNull ExternalTaskError error,
			Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.finishExternalTaskWithError.replace(RestSettings.Params.externalTaskId, externalTaskId), 0, 0, null, null);
	
		HandleExternalTaskErrorRequestPayload body = HandleExternalTaskErrorRequestPayload.builder()
				.workerId(workerId)
				.error(error)
				.build();
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);
		
		return httpResponse.thenApply((HttpResponse<Void> result) -> result.body());
	}

	@Override
	public <TResultType> CompletableFuture<Void> finishExternalTask(@NonNull String workerId, @NonNull String externalTaskId,
			@NonNull TResultType result, Identity identity) {
		List<String> requestAuthHeaders = this.createRequestAuthHeaders(identity);
		
		URI url = this.buildUrl(RestSettings.Paths.finishExternalTask.replace(RestSettings.Params.externalTaskId, externalTaskId), 0, 0, null, null);
		
		FinishExternalTaskRequestPayload<TResultType> body = new FinishExternalTaskRequestPayload<TResultType>();
		body.setWorkerId(workerId);
		body.setResult(result);
		
		CompletableFuture<HttpResponse<Void>> httpResponse = this.httpClient.put(
				url,
				body,
				RequestOptions.builder()
					.headers(requestAuthHeaders)
					.build(),
				Void.class);

		return httpResponse.thenApply((HttpResponse<Void> httpResult) -> httpResult.body());
	}

}
