/*
 * Copyright 2017 Square Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
@file:Suppress("DEPRECATION")

package com.squareup.workflow.legacy.rx2

import com.squareup.workflow.legacy.Reaction
import com.squareup.workflow.legacy.Workflow
import com.squareup.workflow.legacy.WorkflowPool
import com.squareup.workflow.legacy.doLaunch
import io.reactivex.Single
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.channels.ReceiveChannel
import kotlinx.coroutines.rx2.await
import com.squareup.workflow.legacy.Reactor as CoroutineReactor

/**
 * An Rx2 adapter for [com.squareup.workflow.Reactor], allowing implementors to ignore that
 * interface's reliance on [kotlin.coroutines].
 *
 * A [Reactor] is a factory for [Workflow]s implemented as state machines that:
 *
 *  - move through states of [S]
 *  - accept events of type [E]
 *  - emit a result value of type [O] when completed
 *
 * When a new workflow is [launched][launch], each consecutive state will be passed to [onReact],
 * along with an [EventChannel] that can be used to accept events, and a [WorkflowPool] that can be
 * used to delegate work to nested workflows.
 *
 * [onReact] returns a [Single] (read "Future") that eventually emits a [Reaction]
 * indicating what to do next, one of:
 *
 *   - [EnterState][com.squareup.workflow.EnterState]:
 *     Emit [EnterState.state][[com.squareup.workflow.EnterState.state] from [Workflow.state], and
 *     pass it as the next state value of a new call to [onReact].
 *
 *   - [FinishWith][com.squareup.workflow.FinishWith]: Fire [Workflow.result] with
 *     [FinishWith.result][com.squareup.workflow.FinishWith.result], and terminate the workflow.
 *
 * ## Handling Inputs
 *
 * ### Events
 *
 * To handle events received from [Workflow.sendEvent], call [`events.select`][EventChannel.select]
 * from within your [onReact] method:
 *
 *    override fun onReact(
 *      state: MyState,
 *      events: EventChannel<MyEvent>,
 *      workflows: WorkflowPool
 *    ): Single<Reaction<MyState, MyResult>> = when(state) {
 *      FooOrDone -> events.select(state) {
 *        onEvent<Foo> { handleFoo() }
 *        onEvent<Done> { FinishWith(it.result) }
 *      }
 *
 *      FooOrBarState -> events.select(state) {
 *        onEvent<Foo> { handleFoo() }
 *        onEvent<Bar> { EnterState(FooOrDone) }
 *      }
 *    }
 *
 * ### `Single`s
 *
 * [onReact] is not limited to using the given [EventChannel] to calculate its next state. For
 * example, a service call might be handled this way, mapping a [Single] generated by Retrofit to
 * the appropriate [Reaction].
 *
 *    override fun onReact(
 *      state: MyState,
 *      events: EventChannel<MyEvent>,
 *      workflows: WorkflowPool
 *    ): Single<Reaction<MyState, MyResult>> = when(state) {
 *      WaitingForStatusResponse -> statusService.update().map { response ->
 *        if (response.success) EnterState(ShowingStatus(response))
 *        else EnterState(ShowingFailure(response)
 *      }
 *
 *      // ...
 *    }
 *
 * ### Combining Events and `Single`s
 *
 * If you need to mix such command-like [Single]s with workflow events, make a
 * [Worker][com.squareup.workflow.Worker] for your `Single` using [singleWorker].
 * [EventChannel] offers [onWorkerResult][EventSelectBuilder.onWorkerResult] in addition to
 * `onEvent`. Remember to call [WorkflowPool.abandonWorkflow] if you leave while the `Worker` is
 * still running!
 *
 *    private val updateWorker = singleWorker { statusService.update }
 *
 *    override fun onReact(
 *      state: MyState,
 *      events: EventChannel<MyEvent>,
 *      workflows: WorkflowPool
 *    ): Single<Reaction<MyState, MyResult>> = when(state) {
 *      WaitingForStatusResponse -> events.select {
 *
 *        workflows.onWorkerResult(updateWorker) { response ->
 *          if (response.success) EnterState(ShowingStatus(response))
 *          else EnterState(ShowingFailure(response)
 *        }
 *
 *        onEvent<Cancel> {
 *          workflows.abandonWorker(updateWorker)
 *          EnterState(ShowingFailure(Canceled())
 *        }
 *      }
 *
 *      // ...
 *    }
 *
 * ### External Hot `Observable`s
 *
 * To monitor external hot observables, which might fire at any time, subscribe to them in the
 * [launch] method, and map their values to events passed to [Workflow.sendEvent]. Use
 * [Workflow.toCompletable][com.squareup.workflow.rx2.toCompletable] to tear down those
 * subscriptions when the workflow ends.
 *
 *     override fun launch(
 *       initialState: MyState,
 *       workflows: WorkflowPool
 *     ) : Workflow<MyState, MyEvent, MyResult> {
 *       val workflow = doLaunch(initialState, workflows)
 *       val subs = CompositeSubscription()
 *       subs += connectivityMonitor.connectivity.subscribe {
 *         workflow.sendEvent(ConnectivityUpdate(it))
 *       }
 *       subs += workflow.toCompletable().subscribe { subs.clear() }
 *
 *       return workflow
 *     }
 *
 * ## Nesting Workflows
 *
 * To define a state that delegates to a nested workflow, have the [S] subtype that represents it
 * implement [com.squareup.workflow.Delegating]. Use
 * [onNextDelegateReaction][EventSelectBuilder.onWorkflowUpdate] when entering that state to
 * drive the nested workflow and react to its result.
 *
 * For example, in the simplest case, where the parent workflow accepts no events of its own while
 * the delegate is running, the delegating state type would look like this:
 *
 *     data class RunningNested(
 *       // Initial state of the nested workflow, and updated as the
 *       override val delegateState: NestedState = NestedState.start()
 *     ) : MyState(), Delegating<NestedState, NestedEvent, NestedResult> {
 *       override val id = NestedReactor::class.makeWorkflowId()
 *     }
 *
 * You'd register a `NestedReactor` instance with the [WorkflowPool] passed to your [launch]
 * implementation:
 *
 *    class MyReactor(
 *      private val nestedReactor : NestedReactor
 *    ) {
 *      override fun launch(
 *        initialState: MyState,
 *        workflows: WorkflowPool
 *      ) : Workflow<MyState, MyEvent, MyResult> {
 *        workflows.register(nestedReactor)
 *        return doLaunch(initialState, workflows)
 *      }
 *
 * and in your [onReact] method, use
 * [onNextDelegateReaction][EventSelectBuilder.onWorkflowUpdate] to wait for the nested
 * workflow to do its job:
 *
 *    is Delegating -> events.select {
 *      workflows.onNextDelegateReaction(state) {
 *        when (it) {
 *          is EnterState -> EnterState(state.copy(delegateState = it.state))
 *          is FinishWith -> when (it.result) {
 *            is DoSomething -> EnterState(DoingSomething)
 *            is DoSomethingElse -> EnterState(DoingSomethingElse)
 *          }
 *        }
 *      }
 *    }
 *
 * If you need to handle other events while the workflow is running, remember to call
 * [WorkflowPool.abandonWorkflow] if you leave while the nested workflow is still running!
 *
 *    is Delegating -> events.select {
 *      workflows.onNextDelegateReaction(state) {
 *        when (it) {
 *          is EnterState -> EnterState(state.copy(delegateState = it.state))
 *          if FinishWith -> when (it.result) {
 *            is DoSomething -> EnterState(DoingSomething)
 *            is DoSomethingElse -> EnterState(DoingSomethingElse)
 *          }
 *        }
 *      }
 *
 *      onEvent<Cancel> {
 *        workflows.abandonDelegate(state.id)
 *        EnterState(NeverMind)
 *      }
 *    }
 *
 * To accept events for nested workflows, e.g. to drive a UI, define
 * [com.squareup.workflow.Renderer]s for both [S] and each of its
 * [Delegating][com.squareup.workflow.Delegating] subtypes. [WorkflowPool.input] can be used by
 * renderers to route events to any running workflow.
 */
@Deprecated("Use com.squareup.workflow.Workflow")
interface Reactor<S : Any, E : Any, out O : Any> : WorkflowPool.Launcher<S, E, O> {
  fun onReact(
    state: S,
    events: EventChannel<E>,
    workflows: WorkflowPool
  ): Single<out Reaction<S, O>>

  override fun launch(
    initialState: S,
    workflows: WorkflowPool
  ): Workflow<S, E, O> = doLaunch(initialState, workflows)
}

/**
 * Use this to implement [WorkflowPool.Launcher.launch].
 */
@Deprecated("Use com.squareup.workflow.Workflow")
fun <S : Any, E : Any, O : Any> Reactor<S, E, O>.doLaunch(
  initialState: S,
  workflows: WorkflowPool,
  name: String? = null
): Workflow<S, E, O> = toCoroutineReactor().doLaunch(
    initialState, workflows,
    context = CoroutineName(getWorkflowCoroutineName(name))
)

/**
 * Adapter to convert a [Reactor] to a [Reactor].
 */
@Deprecated("Use com.squareup.workflow.Workflow")
fun <S : Any, E : Any, O : Any> Reactor<S, E, O>.toCoroutineReactor() =
  object : CoroutineReactor<S, E, O> {
    override suspend fun onReact(
      state: S,
      events: ReceiveChannel<E>,
      workflows: WorkflowPool
    ): Reaction<S, O> = this@toCoroutineReactor.onReact(state, events.asEventChannel(), workflows)
        .await()

    override fun launch(
      initialState: S,
      workflows: WorkflowPool
    ): Workflow<S, E, O> = this@toCoroutineReactor.launch(initialState, workflows)

    override fun toString(): String =
      "rx2.Reactor(${this@toCoroutineReactor.javaClass.name})"
  }

private fun Reactor<*, *, *>.getWorkflowCoroutineName(name: String?) =
  "workflow(${name ?: this::class.java.name})"
