//   The contents of this file are subject to the Mozilla Public License
//   Version 1.1 (the "License"); you may not use this file except in
//   compliance with the License. You may obtain a copy of the License at
//   http://www.mozilla.org/MPL/
//
//   Software distributed under the License is distributed on an "AS IS"
//   basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
//   License for the specific language governing rights and limitations
//   under the License.
//
//   The Original Code is RabbitMQ.
//
//   The Initial Developers of the Original Code are LShift Ltd.,
//   Cohesive Financial Technologies LLC., and Rabbit Technologies Ltd.
//
//   Portions created by LShift Ltd., Cohesive Financial Technologies
//   LLC., and Rabbit Technologies Ltd. are Copyright (C) 2007-2008
//   LShift Ltd., Cohesive Financial Technologies LLC., and Rabbit
//   Technologies Ltd.;
//
//   All Rights Reserved.
//
//   Contributor(s): ______________________________________.
//

package com.rabbitmq.client.impl;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.IOUtils;

import com.rabbitmq.client.ContentHeader;

/**
 * Generates an AMQP wire-protocol packet from a {@link ContentHeader}. Methods on this object are usually called from autogenerated code.
 */
public class ContentHeaderPropertyWriter {
    /** DataOutputStream wrapper for outBytes */
    public DataOutputStream out;

    /** Current flags word being accumulated */
    public int flagWord;

    /** Position within current flags word */
    public int bitCount;

    /**
     * Constructs a fresh ContentHeaderPropertyWriter.
     */
    public ContentHeaderPropertyWriter(DataOutputStream out) {
        this.out = out;
        this.flagWord = 0;
        this.bitCount = 0;
    }

    public void emitFlagWord(boolean continuationBit) throws IOException {
        out.writeShort(continuationBit ? (flagWord | 1) : flagWord);
        flagWord = 0;
        bitCount = 0;
    }

    public void writePresence(boolean present) throws IOException {
        if (bitCount == 15) {
            emitFlagWord(true);
        }

        if (present) {
            int bit = 15 - bitCount;
            flagWord = flagWord | (1 << bit);
        }
        bitCount++;
    }

    public void finishPresence() throws IOException {
        emitFlagWord(false);
    }

    public void writeShortstr(String str) throws IOException {
        byte[] bytes = str.getBytes("utf-8");
        out.writeByte(bytes.length);
        out.write(bytes);
    }

    public void writeLongstr(String str) throws IOException {
        byte[] bytes = str.getBytes("utf-8");
        out.writeInt(bytes.length);
        out.write(bytes);
    }

    public void writeLongstr(LongString str) throws IOException {
        out.writeInt((int) str.length());
        IOUtils.copy(str.getStream(), out);
    }

    public void writeShort(Integer s) throws IOException {
        out.writeShort(s);
    }

    public void writeLong(Integer l) throws IOException {
        out.writeInt(l);
    }

    public void writeLonglong(Long ll) throws IOException {
        out.writeLong(ll.longValue());
    }

    public void writeTable(Map<String, Object> table) throws IOException {
        out.writeInt((int) Frame.tableSize(table));
        for (Map.Entry<String, Object> entry : table.entrySet()) {

            writeShortstr(entry.getKey());

            Object value = entry.getValue();
            if (value instanceof String) {
                out.writeByte('S');
                writeLongstr((String) value);
            } else if (value instanceof LongString) {
                out.writeByte('S');
                writeLongstr((LongString) value);
            } else if (value instanceof Integer) {
                out.writeByte('I');
                writeLong((Integer) value);
            } else if (value instanceof BigDecimal) {
                out.writeByte('D');
                BigDecimal decimal = (BigDecimal) value;
                out.writeByte(decimal.scale());
                BigInteger unscaled = decimal.unscaledValue();
                if (unscaled.bitLength() > 32) /* Integer.SIZE in Java 1.5 */
                    throw new IllegalArgumentException("BigDecimal too large to be encoded");
                out.writeInt(decimal.unscaledValue().intValue());
            } else if (value instanceof Date) {
                out.writeByte('T');
                writeTimestamp((Date) value);
            } else if (value instanceof Map) {
                out.writeByte('F');
                writeTable((Map<String, Object>) value);
            } else {
                throw new IllegalArgumentException("Invalid value type: " + value.getClass().getName() + " for key " + entry.getKey());
            }
        }
    }

    public void writeOctet(Integer octet) throws IOException {
        out.writeByte(octet);
    }

    public void writeTimestamp(Date timestamp) throws IOException {
        out.writeLong(timestamp.getTime() / 1000);
    }
}
