/*
 * File: DefaultSubmission.java
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * The contents of this file are subject to the terms and conditions of 
 * the Common Development and Distribution License 1.0 (the "License").
 *
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the License by consulting the LICENSE.txt file
 * distributed with this file, or by consulting https://oss.oracle.com/licenses/CDDL
 *
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file LICENSE.txt.
 *
 * MODIFICATIONS:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 */

package com.oracle.coherence.patterns.processing.internal;

import com.oracle.coherence.common.identifiers.Identifier;
import com.oracle.coherence.common.util.ChangeIndication;
import com.oracle.coherence.patterns.processing.SubmissionConfiguration;
import com.oracle.coherence.patterns.processing.SubmissionState;
import com.oracle.coherence.patterns.processing.dispatchers.DispatchController;
import com.tangosol.io.ExternalizableLite;
import com.tangosol.io.pof.PofReader;
import com.tangosol.io.pof.PofWriter;
import com.tangosol.io.pof.PortableObject;
import com.tangosol.util.BinaryEntry;
import com.tangosol.util.ExternalizableHelper;
import com.tangosol.util.UUID;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A {@link DefaultSubmission} captures a request to process some specific payload with associated meta data.
 * <p>
 * Copyright (c) 2009. All Rights Reserved. Oracle Corporation.<br>
 * Oracle is a registered trademark of Oracle Corporation and/or its affiliates.
 *
 * @author Noah Arliss
 * @author Brian Oliver
 * @author Christer Fahlgren
 */
@SuppressWarnings("serial")
public class DefaultSubmission implements PortableObject, ExternalizableLite,
        Submission, ChangeIndication
{
    /**
     * The {@link DispatchController} that accepts or rejects the {@link DefaultSubmission}. This is a per class
     * instance.
     */
    private static DispatchController controller;

    /**
     * The {@link Logger} to use.
     */
    private static final Logger logger = Logger.getLogger(DefaultSubmission.class.getName());

    /**
     * The name of the Coherence Cache that will store {@link DefaultSubmission}s.
     */
    public static final String CACHENAME = "coherence.patterns.processing.submissions";

    /**
     * The {@link SubmissionContent} to process.
     */
    private SubmissionContent payload;

    /**
     * The identifier of the result in the results cache (if any is required / produced).
     */
    private Identifier resultIdentifier;

    /**
     * The {@link UUID} of the submission (generated by a
     * {@link com.oracle.coherence.patterns.processing.ProcessingSession}).
     */
    private UUID submissionUUID;

    /**
     * The {@link Identifier} of the {@link com.oracle.coherence.patterns.processing.ProcessingSession} that submitted
     * this {@link DefaultSubmission}.
     */
    private Identifier sessionIdentifier;

    /**
     * used to keep track of changes related to the {@link ChangeIndication} implementation.
     */
    private transient boolean changed;


    /**
     * Required for {@link ExternalizableLite} and {@link PortableObject}.
     */
    public DefaultSubmission()
    {
    }


    /**
     * Standard Constructor.
     *
     * @param submissionUUID    the {@link UUID} of the submission.
     * @param payload           the object to process in the grid.
     * @param configurationData the {@link DefaultSubmissionConfiguration} associated with the payload
     * @param resultIdentifier  the identifier of the result associated with this {@link DefaultSubmission}
     * @param sessionId         the {@link Identifier} for the
     *        {@link com.oracle.coherence.patterns.processing.ProcessingSession} of this {@link DefaultSubmission}
     */
    public DefaultSubmission(final UUID                    submissionUUID,
                             final Object                  payload,
                             final SubmissionConfiguration configurationData,
                             final Identifier              resultIdentifier,
                             final Identifier              sessionId)
    {
        this.submissionUUID    = submissionUUID;
        this.payload           = new DefaultSubmissionContent(payload, configurationData);
        this.resultIdentifier  = resultIdentifier;
        this.sessionIdentifier = sessionId;
    }


    /**
     * Sets the {@link DispatchController} to be used for this class.
     *
     * @param dispatchController the {@link DispatchController} to use
     */
    public static void setDispatchController(DispatchController dispatchController)
    {
        synchronized (DefaultSubmission.class)
        {
            controller = dispatchController;
        }

    }


    /**
     * Gets the {@link DispatchController} for this class.
     *
     * @return the {@link DispatchController} to use
     */
    public static DispatchController getDispatchController()
    {
        synchronized (DefaultSubmission.class)
        {
            if (controller == null)
            {
                throw new RuntimeException("DispatchController not injected for DefaultSubmission");
            }

            return controller;
        }
    }


    /**
     * {@inheritDoc}
     */
    public SubmissionKey generateKey()
    {
        return new SubmissionKey(payload.getSubmissionConfiguration().getGroupAffinity(), getUUID());
    }


    /**
     * {@inheritDoc}
     */
    public SubmissionContent getContent()
    {
        return payload;
    }


    /**
     * {@inheritDoc}
     */
    public Identifier getResultIdentifier()
    {
        return resultIdentifier;
    }


    /**
     * {@inheritDoc}
     */
    public Identifier getSessionIdentifier()
    {
        return sessionIdentifier;
    }


    /**
     * {@inheritDoc}
     */
    public void changeSessionIdentifier(Identifier newIdentifier)
    {
        if (logger.isLoggable(Level.FINER))
        {
            logger.log(Level.FINER, " Changing sessionIdentifier from:" + sessionIdentifier + " to:" + newIdentifier);
        }

        sessionIdentifier = newIdentifier;
        setChanged();
    }


    /**
     * {@inheritDoc}
     */
    public UUID getUUID()
    {
        return submissionUUID;
    }


    /**
     * {@inheritDoc}
     */
    public boolean reDispatch(long delay)
    {
        // Dispatching must always happen on the node where the submission is stored
        // Otherwise the "restart" mechanism on failure won't kick in.
        getDispatchController().accept(new DefaultPendingSubmission(generateKey(),
                                                                    resultIdentifier,
                                                                    payload.getSubmissionConfiguration(),
                                                                    delay));

        return true;
    }


    /**
     * {@inheritDoc}
     */
    @Override
    public String toString()
    {
        return String.format("%s{uuid=%s, payload=%s, requestdata=%s, resultIdentifier=%s}",
                             this.getClass().getName(),
                             submissionUUID,
                             payload,
                             payload.getSubmissionConfiguration(),
                             resultIdentifier);
    }

    /**
     * {@inheritDoc}
     */
    public void onDeparting(BinaryEntry entry)
    {
        getDispatchController()
                .discard(new DefaultPendingSubmission((SubmissionKey) entry.getKey(),
                        getResultIdentifier(),
                        getContent().getSubmissionConfiguration(),
                        0));
    }

    /**
     * {@inheritDoc}
     */
    public void onInserted(BinaryEntry entry)
    {
        if (logger.isLoggable(Level.FINEST))
        {
            logger.log(Level.FINEST, " {0} was received for dispatch", this);
        }

        getDispatchController().accept(
                new DefaultPendingSubmission(
                        (SubmissionKey) entry.getKey(),
                        getResultIdentifier(),
                        getContent().getSubmissionConfiguration(),
                        getContent().getSubmissionConfiguration()
                                .getSubmissionDelay()));
        /*

        else if (event instanceof BackingMapEntryEvictedEvent)
        {
            throw new RuntimeException("The processing pattern doesn't handle eviction.");
        }
          */
    }

    /**
     * {@inheritDoc}
     */
    public void onArrived(BinaryEntry entry)
    {
        if (logger.isLoggable(Level.FINEST))
        {
            logger.log(Level.FINEST, " {0} was received for dispatch", this);
        }

        try
        {
            SubmissionState state = getDispatchController()
                    .getSubmissionState(resultIdentifier);

            // If a Submission is in a non-final state then we shall re-dispatch it in this JVM
            if (!state.isFinalState())
            {
                getDispatchController()
                        .acceptTransferredSubmission(
                                new DefaultPendingSubmission(
                                        (SubmissionKey) entry.getKey(),
                                        getResultIdentifier(),
                                        getContent()
                                                .getSubmissionConfiguration(),
                                        getContent()
                                                .getSubmissionConfiguration()
                                                .getSubmissionDelay()));
            }
            else
            {
                if (logger.isLoggable(Level.FINE))
                {
                    logger.log(Level.FINE,
                            "Submission {0} was failed over and is in a final state.",
                            this);
                }
            }
        }
        catch (Exception e)
        {
            if (logger.isLoggable(Level.WARNING))
            {
                logger.log(Level.WARNING,
                        "Submission {0} was failed over but is already removed. Exception caught: {1}.",
                        new Object[]{this, e});
            }

        }
    }

    /*

  todo PFM - what to do here?

  else if (event instanceof BackingMapEntryEvictedEvent)
  {
      throw new RuntimeException("The processing pattern doesn't handle eviction.");
  }
    */


    /**
     * {@inheritDoc}
     */
    public void beforeChange()
    {
        changed = false;
    }


    /**
     * {@inheritDoc}
     */
    public boolean changed()
    {
        return changed;
    }


    /**
     * Sets the state to be changed for the benefit of the {@link ChangeIndication} implementation.
     */
    private void setChanged()
    {
        changed = true;
    }


    /**
     * {@inheritDoc}
     */
    public void readExternal(final DataInput in) throws IOException
    {
        this.submissionUUID    = (UUID) ExternalizableHelper.readObject(in);
        this.payload           = (SubmissionContent) ExternalizableHelper.readObject(in);
        this.resultIdentifier  = (Identifier) ExternalizableHelper.readObject(in);
        this.sessionIdentifier = (Identifier) ExternalizableHelper.readObject(in);
    }


    /**
     * {@inheritDoc}
     */
    public void writeExternal(final DataOutput out) throws IOException
    {
        ExternalizableHelper.writeObject(out, this.submissionUUID);
        ExternalizableHelper.writeObject(out, this.payload);
        ExternalizableHelper.writeObject(out, this.resultIdentifier);
        ExternalizableHelper.writeObject(out, this.sessionIdentifier);
    }


    /**
     * {@inheritDoc}
     */
    public void readExternal(final PofReader reader) throws IOException
    {
        this.submissionUUID    = (UUID) reader.readObject(0);
        this.payload           = (SubmissionContent) reader.readObject(1);
        this.resultIdentifier  = (Identifier) reader.readObject(2);
        this.sessionIdentifier = (Identifier) reader.readObject(3);
    }


    /**
     * {@inheritDoc}
     */
    public void writeExternal(final PofWriter writer) throws IOException
    {
        writer.writeObject(0, this.submissionUUID);
        writer.writeObject(1, this.payload);
        writer.writeObject(2, this.resultIdentifier);
        writer.writeObject(3, this.sessionIdentifier);
    }
}
