/*
 * Copyright (c) 2020, 2021, Oracle and/or its affiliates.
 *
 * Licensed under the Universal Permissive License v 1.0 as shown at
 * http://oss.oracle.com/licenses/upl.
 */
package com.oracle.coherence.concurrent.atomic;

import com.tangosol.io.pof.PofReader;
import com.tangosol.io.pof.PofSerializer;
import com.tangosol.io.pof.PofWriter;

import java.io.IOException;

/**
 * An {@code AtomicStampedReference} maintains an object reference
 * along with an integer "stamp", that can be updated atomically.
 *
 * @param <V>  the type of object referred to by this reference
 *
 * @author Aleks Seovic  2020.12.09
 * @since 21.12
 */
public interface AtomicStampedReference<V>
    {
    /**
     * Return non-blocking API for this atomic reference.
     *
     * @return non-blocking API for this atomic reference
     */
    AsyncAtomicStampedReference<V> async();

    /**
     * Returns the current value of the reference.
     *
     * @return the current value of the reference
     */
    V getReference();

    /**
     * Returns the current value of the stamp.
     *
     * @return the current value of the stamp
     */
    int getStamp();

    /**
     * Returns the current values of both the reference and the stamp.
     * Typical usage is {@code int[1] holder; ref = v.get(holder); }.
     *
     * @param iaStampHolder  an array of size of at least one.  On return,
     *                       {@code stampHolder[0]} will hold the value of the stamp.
    *
     * @return the current value of the reference
     */
    V get(int[] iaStampHolder);

    /**
     * Atomically sets the value of both the reference and stamp
     * to the given update values if the
     * current reference is {@code ==} to the expected reference
     * and the current stamp is equal to the expected stamp.
     *
     * @param expectedReference  the expected value of the reference
     * @param newReference       the new value for the reference
     * @param nExpectedStamp     the expected value of the stamp
     * @param nNewStamp          the new value for the stamp

     * @return {@code true} if successful
     */
    boolean compareAndSet(V expectedReference, V newReference, int nExpectedStamp, int nNewStamp);

    /**
     * Unconditionally sets the value of both the reference and stamp.
     *
     * @param newReference  the new value for the reference
     * @param nNewStamp     the new value for the stamp
     */
    void set(V newReference, int nNewStamp);

    /**
     * Atomically sets the value of the stamp to the given update value
     * if the current reference is {@code ==} to the expected
     * reference.  Any given invocation of this operation may fail
     * (return {@code false}) spuriously, but repeated invocation
     * when the current value holds the expected value and no other
     * thread is also attempting to set the value will eventually
     * succeed.
     *
     * @param expectedReference  the expected value of the reference
     * @param nNewStamp           the new value for the stamp
     *
     * @return {@code true} if successful
     */
    boolean attemptStamp(V expectedReference, int nNewStamp);

    // ----- inner class: Serializer ----------------------------------------

    /**
     * POF serializer implementation.
     *
     * @param <V>  the type of object referred to by this reference
     */
    class Serializer<V>
            implements PofSerializer<java.util.concurrent.atomic.AtomicStampedReference<V>>
        {
        // ----- PofSerializer interface ------------------------------------

        @Override
        public void serialize(PofWriter out, java.util.concurrent.atomic.AtomicStampedReference<V> value)
                throws IOException
            {
            int[] aiStamp = new int[1];

            out.writeObject(0, value.get(aiStamp));
            out.writeInt(1, aiStamp[0]);
            out.writeRemainder(null);
            }

        @Override
        public java.util.concurrent.atomic.AtomicStampedReference<V> deserialize(PofReader in)
                throws IOException
            {
            V   value  = in.readObject(0);
            int nStamp = in.readInt(1);

            in.readRemainder();
            return new java.util.concurrent.atomic.AtomicStampedReference<>(value, nStamp);
            }
        }
    }
