package com.nimbusds.openid.connect.provider.spi.tokens.response.customizer;


import java.util.Collections;
import java.util.List;

import com.thetransactioncompany.util.PropertyParseException;
import com.thetransactioncompany.util.PropertyRetriever;
import net.minidev.json.JSONObject;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.checkerframework.checker.nullness.qual.Nullable;

import com.nimbusds.common.config.ConfigurationException;
import com.nimbusds.common.config.LoggableConfiguration;
import com.nimbusds.oauth2.sdk.AccessTokenResponse;
import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.TokenErrorResponse;
import com.nimbusds.oauth2.sdk.TokenResponse;
import com.nimbusds.oauth2.sdk.util.MapUtils;
import com.nimbusds.openid.connect.provider.spi.tokens.response.CustomTokenResponseComposer;
import com.nimbusds.openid.connect.provider.spi.tokens.response.TokenResponseContext;


/**
 * Token response customiser.
 *
 * <p>Example configuration:
 *
 * <pre>
 * op.token.response.customizer.enable=true
 * op.token.response.customizer.includeAuthzDataParams=vp_token,authorization_details
 * </pre>
 */
public class TokenResponseCustomizer implements CustomTokenResponseComposer, LoggableConfiguration {
	
	
	/**
	 * The Connect2id server MAIN logger.
	 */
	protected static final Logger mainLog = LogManager.getLogger("MAIN");
	
	
	/**
	 * The Connect2id server token endpoint logger.
	 */
	protected static final Logger tokenEndpointLog = LogManager.getLogger("TOKEN");
	
	
	/**
	 * {@code true} if the customiser is enabled.
	 */
	public final boolean enable;
	
	
	/**
	 * The list of authorisation data parameters to include.
	 */
	public final List<String> paramNames;
	
	
	/**
	 * Creates a new token response customiser configured from system
	 * properties.
	 */
	public TokenResponseCustomizer() {
	
		var pr = new PropertyRetriever(System.getProperties());
		
		try {
			enable = pr.getOptBoolean("op.token.response.customizer.enable", false);
			
			if (! enable) {
				paramNames = Collections.emptyList();
			} else {
				paramNames = pr.getOptStringList("op.token.response.customizer.includeAuthzDataParams", Collections.emptyList());
			}
			
		} catch (PropertyParseException e) {
			throw new ConfigurationException(e.getMessage());
		}
		
		log();
	}
	
	
	@Override
	public void log() {
		
		mainLog.info("[TRC0000] Token response customizer enabled: {}", enable);
		
		if (! enable) {
			return;
		}
		
		mainLog.info("[TRC0001] Token response customizer: Included authorization data parameters: {}", paramNames);
	}
	
	
	protected JSONObject collectCustomParams(final @Nullable JSONObject authzData) {
		
		var out = new JSONObject();
		
		if (MapUtils.isEmpty(authzData)) {
			return out;
		}
		
		for (String name: paramNames) {
			if (authzData.get(name) != null) {
				out.put(name, authzData.get(name));
			}
		}
		
		return out;
	}
	
	
	@Override
	public TokenResponse compose(final TokenResponse originalResponse,
				     final TokenResponseContext context) {
	
		if (! enable) {
			return originalResponse;
		}
		
		if (originalResponse instanceof TokenErrorResponse) {
			return originalResponse;
		}
		
		AccessTokenResponse successResponse = originalResponse.toSuccessResponse();
		
		JSONObject authzData = context.getAuthorizationData();
		
		if (MapUtils.isEmpty(authzData)) {
			return originalResponse; // No data
		}
		
		try {
			JSONObject tokensObject = successResponse.toJSONObject();
			
			JSONObject customParams = collectCustomParams(authzData);
			
			tokenEndpointLog.debug("[TRC0002] Token response customizer: Included params: {}", customParams.keySet());
			
			tokensObject.putAll(customParams);
			
			return AccessTokenResponse.parse(tokensObject);
			
		} catch (ParseException e) {
			throw new RuntimeException("Internal error: " + e.getMessage(), e);
		}
	}
}
