package com.nimbusds.openid.connect.provider.spi.grants.password.webapi;


import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.nimbusds.oauth2.sdk.*;
import com.nimbusds.oauth2.sdk.http.CommonContentTypes;
import com.nimbusds.oauth2.sdk.http.HTTPRequest;
import com.nimbusds.oauth2.sdk.http.HTTPResponse;
import com.nimbusds.oauth2.sdk.id.ClientID;

import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;

import com.nimbusds.openid.connect.provider.spi.InitContext;
import com.nimbusds.openid.connect.provider.spi.grants.PasswordGrantAuthorization;
import com.nimbusds.openid.connect.provider.spi.grants.PasswordGrantHandler;


/**
 * Password grant web API.
 */
public class PasswordGrantWebAPI implements PasswordGrantHandler {


	/**
	 * The configuration file path.
	 */
	public static final String CONFIG_FILE_PATH = "/WEB-INF/passwordGrantHandlerWebAPI.properties";


	/**
	 * The configuration.
	 */
	private Configuration config;


	/**
	 * The main logger.
	 */
	private static final Logger mainLog = LogManager.getLogger("MAIN");


	/**
	 * The token endpoint logger.
	 */
	private static final Logger tokenEndpointLog = LogManager.getLogger("TOKEN");


	/**
	 * Loads the configuration.
	 *
	 * @param initContext The initialisation context. Must not be
	 *                    {@code null}.
	 *
	 * @return The configuration.
	 *
	 * @throws Exception If loading failed.
	 */
	private static Configuration loadConfiguration(final InitContext initContext)
		throws Exception {

		InputStream inputStream = initContext.getResourceAsStream(CONFIG_FILE_PATH);

		if (inputStream == null) {
			throw new Exception("Couldn't find password grant handler configuration file: " + CONFIG_FILE_PATH);
		}

		Properties props = new Properties();
		props.load(inputStream);

		// Override with any system properties
		logOverridingSystemProperties();
		props.putAll(System.getProperties());

		return new Configuration(props);
	}


	/**
	 * Logs the overriding system properties.
	 */
	public static void logOverridingSystemProperties() {

		Properties sysProps = System.getProperties();

		StringBuilder sb = new StringBuilder();

		for (String key: sysProps.stringPropertyNames()) {

			if (! key.startsWith(Configuration.DEFAULT_PREFIX))
				continue;

			if (sb.length() > 0)
				sb.append(" ");

			sb.append(key);
		}

		mainLog.info("[PGA 0004] Overriding system properties: {}", sb);
	}


	@Override
	public void init(final InitContext initContext)
		throws Exception {

		mainLog.info("[PGA 0005] Initializing password grant handler...");
		config = loadConfiguration(initContext);
		config.log();
	}


	/**
	 * Returns the configuration.
	 *
	 * @return The configuration.
	 */
	public Configuration getConfiguration() {

		return config;
	}


	@Override
	public boolean isEnabled() {

		return config.enable;
	}


	@Override
	public GrantType getGrantType() {

		return GrantType.PASSWORD;
	}


	@Override
	public PasswordGrantAuthorization processGrant(final ResourceOwnerPasswordCredentialsGrant grant,
						       final Scope scope,
						       final ClientID clientID,
						       final boolean confidentialClient,
						       final OIDCClientMetadata clientMetadata)
		throws GeneralException {

		if (! config.enable) {
			throw new GeneralException("Grant handler disabled", OAuth2Error.UNSUPPORTED_GRANT_TYPE);
		}

		tokenEndpointLog.debug("[PGA 0006] Password grant handler: Received request with username={} password=[hidden] scope={}",
			grant.getUsername(), scope);

		HandlerRequest request = new HandlerRequest(grant, scope, clientID, confidentialClient, clientMetadata);

		HTTPRequest httpRequest = new HTTPRequest(HTTPRequest.Method.POST, config.url);
		httpRequest.setAuthorization(config.apiAccessToken.toAuthorizationHeader());
		httpRequest.setContentType(CommonContentTypes.APPLICATION_JSON);
		httpRequest.setQuery(request.toJSONObject().toJSONString());
		httpRequest.setConnectTimeout(config.connectTimeout);
		httpRequest.setReadTimeout(config.readTimeout);

		tokenEndpointLog.debug("[PGA 0007] Password grant handler: Making HTTP post request to {}", httpRequest.getURL());

		HTTPResponse httpResponse;

		try {
			httpResponse = httpRequest.send();

		} catch (IOException e) {
			// Log and return HTTP 500 server_error back to OAuth 2.0 client
			tokenEndpointLog.error("[PGA 0008] Password grant handler: HTTP exception: " + e.getMessage(), e);
			throw new GeneralException("Server error", OAuth2Error.SERVER_ERROR);
		}

		if (! httpResponse.indicatesSuccess()) {
			ErrorObject errorObject = processNon200Response(httpResponse);
			throw new GeneralException(errorObject.getCode(), errorObject);
		}

		tokenEndpointLog.debug("[PGA 0009] Password grant handler: Received authorization response: {}", httpResponse.getContent());

		try {
			return PasswordGrantAuthorization.parse(httpResponse.getContentAsJSONObject());

		} catch (Exception e) {
			// Log and pass HTTP 500 server_error back to OAuth 2.0 client
			tokenEndpointLog.error("[PGA 0010] Password grant handler: Invalid authorization response: {}", e.getMessage(), e);
			throw new GeneralException("Server error", OAuth2Error.SERVER_ERROR);
		}
	}


	/**
	 * Processes a non-200 HTTP response to produce an OAuth 2.0 error
	 * object to return to the OAuth 2.0 client.
	 *
	 * @param httpResponse The HTTP response, with a status code other than
	 *                     200. Must not be {@code null}.
	 *
	 * @return The resulting error object.
	 */
	public static ErrorObject processNon200Response(final HTTPResponse httpResponse) {

		// Check for non 400
		if (httpResponse.getStatusCode() != HTTPResponse.SC_BAD_REQUEST) {
			// Unexpected HTTP status code
			tokenEndpointLog.error("[PGA 0011] Password grant handler: Unexpected HTTP response: {}", httpResponse.getStatusCode());
			return OAuth2Error.SERVER_ERROR;
		}

		// Check error code
		ErrorObject errorObject = ErrorObject.parse(httpResponse);

		if (errorObject.getCode() == null ||
		    ! errorObject.equals(OAuth2Error.INVALID_GRANT) &&
		    ! errorObject.equals(OAuth2Error.INVALID_SCOPE)) {

			// Missing or unexpected error code
			tokenEndpointLog.error("[PGA 0012] Password grant handler: Missing or unexpected error code: {}", errorObject.getCode());
			return OAuth2Error.SERVER_ERROR;
		}

		// Return parsed invalid_grant or invalid_scope
		tokenEndpointLog.info("[PGA 0013] Password grant handler: Token request denied: {}", errorObject.getCode());
		return errorObject;
	}


	@Override
	public void shutdown()
		throws Exception {

		// Nothing to do

		mainLog.info("[PGA 0014] Shutting down password grant handler...");
	}
}
