package com.nimbusds.openid.connect.provider.spi.grants.jwt.selfissued.handler;


import java.io.InputStream;
import java.util.HashSet;
import java.util.Properties;
import java.util.Set;

import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.oauth2.sdk.GeneralException;
import com.nimbusds.oauth2.sdk.GrantType;
import com.nimbusds.oauth2.sdk.OAuth2Error;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.id.Subject;
import com.nimbusds.openid.connect.provider.spi.InitContext;
import com.nimbusds.openid.connect.provider.spi.grants.*;
import com.nimbusds.openid.connect.sdk.OIDCScopeValue;
import com.nimbusds.openid.connect.sdk.rp.OIDCClientMetadata;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * Self-issued JWT bearer grant handler. Implements the
 * {@code com.nimbusds.openid.connect.provider.spi.grants.SelfIssuedJWTGrantHandler}
 * Service Provider Interface (SPI) for plugging grant handlers into the
 * Connect2id server. The authorised scope is bounded by the registered scope
 * values in the OAuth 2.0 client metadata.
 *
 * <p>Note that self-issued (by the client) JWT bearer assertions are verified
 * (claims and JWS HMAC / signature) by the Connect2id server prior to calling
 * the handler. The handler only needs to determine the scope and optional
 * parameters for the access token. This handler doesn't support issue of ID
 * tokens.
 *
 * <p>Processing rules:
 *
 * <ol>
 *     <li>If no scope values are registered in the metadata for the requesting
 *         client, an {@code invalid_scope} OAuth 2.0 error is returned. See
 *         RFC 6749, section 5.2.
 *     <li>If no scope is explicitly requested (with the {@code scope} 
 *         parameter of the token request), the authorised scope defaults to  
 *         the registered scope values for the client.
 *     <li>If an explicit scope if requested (with the {@code scope} parameter
 *         of the token request), the authorised scope is reduced to those
 *         scope values for which the client is registered. See RFC 6749, 
 *         section 3.3.
 * </ol>
 *
 * <p>See RFC 7523 for more information on using JWT bearer assertions as OAuth
 * grants.
 *
 * <p>The Connect2id server also supports an SPI for handling JWT bearer
 * assertions issued by a third-party (such as a token service, or another
 * authorisation / identity provider).
 */
public class SimpleSelfIssuedJWTGrantHandler implements SelfIssuedJWTGrantHandler {


	/**
	 * The configuration file path.
	 */
	public static final String CONFIG_FILE_PATH = "/WEB-INF/selfIssuedJWTBearerHandler.properties";


	/**
	 * The configuration.
	 */
	private Configuration config;


	/**
	 * The main logger.
	 */
	private static final Logger MAIN_LOG = LogManager.getLogger("MAIN");


	/**
	 * The token endpoint logger.
	 */
	private static final Logger TOKEN_ENDPOINT_LOG = LogManager.getLogger("TOKEN");


	/**
	 * Loads the configuration.
	 *
	 * @param initContext The initialisation context. Must not be
	 *                    {@code null}.
	 *
	 * @return The configuration.
	 *
	 * @throws Exception If loading failed.
	 */
	private static Configuration loadConfiguration(final InitContext initContext)
		throws Exception {

		InputStream inputStream = initContext.getResourceAsStream(CONFIG_FILE_PATH);

		if (inputStream == null) {
			throw new Exception("Couldn't find self-issued JWT bearer grant handler configuration file: " + CONFIG_FILE_PATH);
		}

		Properties props = new Properties();
		props.load(inputStream);

		// Override with any system properties
		logOverridingSystemProperties();
		props.putAll(System.getProperties());

		return new Configuration(props);
	}


	/**
	 * Logs the overriding system properties.
	 */
	public static void logOverridingSystemProperties() {

		Properties sysProps = System.getProperties();

		StringBuilder sb = new StringBuilder();

		for (String key: sysProps.stringPropertyNames()) {

			if (! key.startsWith(Configuration.PREFIX))
				continue;

			if (sb.length() > 0)
				sb.append(" ");

			sb.append(key);
		}

		MAIN_LOG.info("[SJH 0001] Overriding system properties: {}", sb);
	}


	@Override
	public void init(final InitContext initContext)
		throws Exception {

		MAIN_LOG.info("[SJH 0000] Initializing self-issued JWT bearer grant handler...");
		config = loadConfiguration(initContext);
		config.log();
	}


	/**
	 * Returns the configuration.
	 *
	 * @return The configuration.
	 */
	public Configuration getConfiguration() {

		return config;
	}


	@Override
	public GrantType getGrantType() {

		return GrantType.JWT_BEARER;
	}


	@Override
	public boolean isEnabled() {

		return config.enable;
	}


	@Override
	public SelfIssuedAssertionAuthorization processSelfIssuedGrant(final JWTClaimsSet jwtClaimsSet,
								       final Scope scope,
								       final ClientID clientID,
								       final OIDCClientMetadata clientMetadata)
		throws GeneralException {

		TOKEN_ENDPOINT_LOG.debug("[SJH 0002] Self-issued JWT bearer grant handler: Received request from client_id={} with scope={}", clientID, scope);

		final Scope registeredScopeValues = clientMetadata.getScope();

		if (registeredScopeValues == null || registeredScopeValues.isEmpty()) {
			String msg = "No registered scopes for client";
			throw new GeneralException(msg, OAuth2Error.INVALID_SCOPE.setDescription(msg));
		}

		Scope authorizedScope;

		if (scope == null || scope.isEmpty()) {
			// Implicit scope request, default to registered scope values
			authorizedScope = registeredScopeValues;
		} else {
			// Explicit scope request, discard any non-registered scope values
			authorizedScope = scope;
			authorizedScope.retainAll(registeredScopeValues);

			if (authorizedScope.isEmpty()) {
				// No scope values match
				String msg = "None of the requested scope values are permitted for this client";
				throw new GeneralException(msg, OAuth2Error.INVALID_SCOPE.setDescription(msg));
			}
		}

		// Compose the authorisation spec
		Subject sub = new Subject(jwtClaimsSet.getSubject());

		AccessTokenSpec accessTokenSpec = new AccessTokenSpec(
			config.accessToken.lifetime,
			config.accessToken.audienceList,
			config.accessToken.encoding,
			config.accessToken.encrypt);

		return new SelfIssuedAssertionAuthorization(
			sub,
			authorizedScope,
			accessTokenSpec,
			IDTokenSpec.NONE,
			resolveOpenIDClaims(authorizedScope),
			null); // Optional data
	}


	/**
	 * Resolves the matching OpenID claims (if any) for the specified
	 * authorised scope.
	 *
	 * @param authorizedScope The authorised scope. May include OpenID
	 *                        standard scope values. Must not be
	 *                        {@code null}.
	 *
	 * @return The claims spec.
	 */
	public static ClaimsSpec resolveOpenIDClaims(final Scope authorizedScope) {

		Set<String> claimNames = new HashSet<>();

		if (authorizedScope.contains(OIDCScopeValue.EMAIL)) {
			claimNames.addAll(OIDCScopeValue.EMAIL.getClaimNames());
		}

		if (authorizedScope.contains(OIDCScopeValue.PHONE)) {
			claimNames.addAll(OIDCScopeValue.PHONE.getClaimNames());
		}

		if (authorizedScope.contains(OIDCScopeValue.PROFILE)) {
			claimNames.addAll(OIDCScopeValue.PROFILE.getClaimNames());
		}

		if (authorizedScope.contains(OIDCScopeValue.ADDRESS)) {
			claimNames.addAll(OIDCScopeValue.ADDRESS.getClaimNames());
		}

		if (! claimNames.isEmpty()) {
			return new ClaimsSpec(claimNames);
		} else {
			return ClaimsSpec.NONE;
		}
	}


	@Override
	public void shutdown()
		throws Exception {

		// Nothing to do
		MAIN_LOG.info("[SJH 0003] Shut down self-issued JWT bearer grant handler");
	}
}
