package com.nimbusds.infinispan.persistence.dynamodb.query;


import java.util.function.Consumer;

import com.amazonaws.services.dynamodbv2.document.Index;
import com.amazonaws.services.dynamodbv2.document.ItemCollection;
import com.amazonaws.services.dynamodbv2.document.QueryOutcome;
import com.amazonaws.services.dynamodbv2.document.spec.QuerySpec;
import com.amazonaws.services.dynamodbv2.document.utils.NameMap;
import com.amazonaws.services.dynamodbv2.document.utils.ValueMap;
import com.nimbusds.infinispan.persistence.common.InfinispanEntry;
import com.nimbusds.infinispan.persistence.common.query.Query;
import com.nimbusds.infinispan.persistence.common.query.SimpleMatchQuery;
import com.nimbusds.infinispan.persistence.common.query.UnsupportedQueryException;
import net.jcip.annotations.ThreadSafe;


/**
 * Simple match DynamoDB query executor. Accepts queries of type
 * {@link SimpleMatchQuery} where both key name and value must be of type
 * {@link String}.
 */
@ThreadSafe
public class SimpleMatchQueryExecutor<K, V> implements DynamoDBQueryExecutor<K, V> {
	
	
	/**
	 * The initialisation context.
	 */
	private DynamoDBQueryExecutorInitContext<K, V> initCtx;
	
	
	@Override
	public void init(final DynamoDBQueryExecutorInitContext<K, V> initCtx) {
		this.initCtx = initCtx;
	}
	
	
	/**
	 * Returns the DynamoDB query spec for the specified simple match
	 * query.
	 *
	 * @param simpleMatchQuery The simple match query. Must not be
	 *                         {@code null}.
	 *
	 * @return The DynamoDB query spec.
	 */
	private static QuerySpec toQuerySpec(final SimpleMatchQuery<String, String> simpleMatchQuery) {
		
		return new QuerySpec()
			.withKeyConditionExpression("#k = :value")
			.withNameMap(new NameMap()
				.with("#k", simpleMatchQuery.getKey()))
			.withValueMap(new ValueMap()
				.withString(":value", simpleMatchQuery.getValue()));
	}
	
	
	@Override
	@SuppressWarnings("unchecked")
	public void executeQuery(final Query query, final Consumer<InfinispanEntry<K, V>> consumer) {
		
		if (! (query instanceof SimpleMatchQuery)) {
			throw new UnsupportedQueryException(query);
		}
		
		SimpleMatchQuery<String, String> simpleMatchQuery = (SimpleMatchQuery<String, String>)query;
		
		QuerySpec querySpec = toQuerySpec(simpleMatchQuery);
		
		Index index = initCtx.getDynamoDBIndex(simpleMatchQuery.getKey());
		
		ItemCollection<QueryOutcome> items = index.query(querySpec);
		
		items.forEach(item -> consumer.accept(initCtx.getDynamoDBItemTransformer().toInfinispanEntry(item)));
	}
}
