/*
 * Decompiled with CFR 0.152.
 */
package io.jaegertracing.internal.reporters;

import io.jaegertracing.internal.JaegerSpan;
import io.jaegertracing.internal.exceptions.SenderException;
import io.jaegertracing.internal.metrics.InMemoryMetricsFactory;
import io.jaegertracing.internal.metrics.Metrics;
import io.jaegertracing.internal.senders.SenderResolver;
import io.jaegertracing.spi.Reporter;
import io.jaegertracing.spi.Sender;
import java.util.HashSet;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RemoteReporter
implements Reporter {
    private static final Logger log = LoggerFactory.getLogger(RemoteReporter.class);
    private static final int DEFAULT_CLOSE_ENQUEUE_TIMEOUT_MILLIS = 1000;
    public static final int DEFAULT_FLUSH_INTERVAL_MS = 1000;
    public static final int DEFAULT_MAX_QUEUE_SIZE = 100;
    private final Sender sender;
    private final int closeEnqueueTimeout;
    private final BlockingQueue<Command> commandQueue;
    private final Timer flushTimer;
    private final Thread queueProcessorThread;
    private final QueueProcessor queueProcessor;
    private final Metrics metrics;

    private RemoteReporter(Sender sender, int flushInterval, int maxQueueSize, int closeEnqueueTimeout, Metrics metrics) {
        this.sender = sender;
        this.metrics = metrics;
        this.closeEnqueueTimeout = closeEnqueueTimeout;
        this.commandQueue = new ArrayBlockingQueue<Command>(maxQueueSize);
        this.queueProcessor = new QueueProcessor();
        this.queueProcessorThread = new Thread((Runnable)this.queueProcessor, "jaeger.RemoteReporter-QueueProcessor");
        this.queueProcessorThread.setDaemon(true);
        this.queueProcessorThread.start();
        this.flushTimer = new Timer("jaeger.RemoteReporter-FlushTimer", true);
        this.flushTimer.schedule(new TimerTask(){

            @Override
            public void run() {
                RemoteReporter.this.flush();
            }
        }, flushInterval, (long)flushInterval);
    }

    @Override
    public void report(JaegerSpan span) {
        boolean added = this.commandQueue.offer(new AppendCommand(span));
        if (!added) {
            this.metrics.reporterDropped.inc(1L);
        }
    }

    @Override
    public void close() {
        try {
            boolean added = this.commandQueue.offer(new CloseCommand(), this.closeEnqueueTimeout, TimeUnit.MILLISECONDS);
            if (added) {
                this.queueProcessorThread.join(10000L);
            } else {
                log.warn("Unable to cleanly close RemoteReporter, command queue is full - probably the sender is stuck");
            }
        }
        catch (InterruptedException e) {
            log.error("Interrupted", e);
        }
        finally {
            try {
                int n = this.sender.close();
                this.metrics.reporterSuccess.inc(n);
            }
            catch (SenderException e) {
                this.metrics.reporterFailure.inc(e.getDroppedSpanCount());
                log.error("Remote reporter error", e);
            }
            this.flushTimer.cancel();
        }
    }

    void flush() {
        this.metrics.reporterQueueLength.update(this.commandQueue.size());
        this.commandQueue.offer(new FlushCommand());
    }

    public String toString() {
        return "RemoteReporter(sender=" + this.sender + ", closeEnqueueTimeout=" + this.closeEnqueueTimeout + ")";
    }

    public static class Builder {
        private Sender sender;
        private int flushInterval = 1000;
        private int maxQueueSize = 100;
        private int closeEnqueTimeout = 1000;
        private Metrics metrics;

        public Builder withFlushInterval(int flushInterval) {
            this.flushInterval = flushInterval;
            return this;
        }

        public Builder withMaxQueueSize(int maxQueueSize) {
            this.maxQueueSize = maxQueueSize;
            return this;
        }

        public Builder withMetrics(Metrics metrics) {
            this.metrics = metrics;
            return this;
        }

        public Builder withSender(Sender sender) {
            this.sender = sender;
            return this;
        }

        public Builder withCloseEnqueueTimeout(int closeEnqueueTimeoutMs) {
            this.closeEnqueTimeout = closeEnqueueTimeoutMs;
            return this;
        }

        public RemoteReporter build() {
            if (this.sender == null) {
                this.sender = SenderResolver.resolve();
            }
            if (this.metrics == null) {
                this.metrics = new Metrics(new InMemoryMetricsFactory());
            }
            return new RemoteReporter(this.sender, this.flushInterval, this.maxQueueSize, this.closeEnqueTimeout, this.metrics);
        }
    }

    class QueueProcessor
    implements Runnable {
        private volatile boolean open = true;
        private final Set<Class<?>> commandFailedBefore = new HashSet();

        QueueProcessor() {
        }

        @Override
        public void run() {
            while (this.open) {
                try {
                    Command command = (Command)RemoteReporter.this.commandQueue.take();
                    Class<?> commandClass = command.getClass();
                    boolean failedBefore = this.commandFailedBefore.contains(commandClass);
                    try {
                        int sentSpanCount = command.execute();
                        if (sentSpanCount <= 0) continue;
                        ((RemoteReporter)RemoteReporter.this).metrics.reporterSuccess.inc(sentSpanCount);
                        if (!failedBefore) continue;
                        log.info(commandClass.getSimpleName() + " is working again!");
                        this.commandFailedBefore.remove(commandClass);
                    }
                    catch (SenderException e) {
                        ((RemoteReporter)RemoteReporter.this).metrics.reporterFailure.inc(e.getDroppedSpanCount());
                        if (failedBefore) continue;
                        log.warn(commandClass.getSimpleName() + " execution failed! Repeated errors of this command will not be logged.", e);
                        this.commandFailedBefore.add(commandClass);
                    }
                }
                catch (Exception e) {
                    log.error("QueueProcessor error:", e);
                }
            }
        }

        public void close() {
            this.open = false;
        }

        public String toString() {
            return "RemoteReporter.QueueProcessor(open=" + this.open + ", commandFailedBefore=" + this.commandFailedBefore + ")";
        }
    }

    class FlushCommand
    implements Command {
        FlushCommand() {
        }

        @Override
        public int execute() throws SenderException {
            return RemoteReporter.this.sender.flush();
        }
    }

    class CloseCommand
    implements Command {
        CloseCommand() {
        }

        @Override
        public int execute() throws SenderException {
            RemoteReporter.this.queueProcessor.close();
            return 0;
        }
    }

    class AppendCommand
    implements Command {
        private final JaegerSpan span;

        public AppendCommand(JaegerSpan span) {
            this.span = span;
        }

        @Override
        public int execute() throws SenderException {
            return RemoteReporter.this.sender.append(this.span);
        }
    }

    static interface Command {
        public int execute() throws SenderException;
    }
}

