// Copyright (c) Microsoft. All rights reserved.

package com.microsoft.trouterclient;

public class TrouterUtils {
    /**
     * Helper function to replace URL base (protocol + hostname)
     *
     * Useful when converting Trouter Endpoint URL to its client-to-client variant
     * with `ITrouterConnectionInfo.c2cUrlBase`.
     *
     * @param existingUrl Original URL
     * @param newBase New URL base
     * @return The original URL with the base replaced
     */
    public static String replaceUrlBase(String existingUrl, String newBase) {
        int schemaSepPos = existingUrl.indexOf("://");
        if (schemaSepPos >= 0) {
            int pathSepPos = existingUrl.indexOf('/', schemaSepPos + 3);
            if (pathSepPos >= 0) {
                return (newBase + existingUrl.substring(pathSepPos));
            }
        }
        return "";
    }

    /**
     * Helper function to validate that string value is a valid hostname
     * It should only contain alphanums, dots and hyphens as per https://tools.ietf.org/html/rfc1123
     *
     * Intended for internal use only thus not public
     */
    static boolean isValidTrouterUrlHostname(String hostname) {
        if (hostname == null || hostname.isEmpty()) {
            return false;
        }

        for (int i = 0; i <  hostname.length(); i++) {
            char ch = hostname.charAt(i);
            if (ch != '.' && ch != '-' && !Character.isLetterOrDigit(ch)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Helper function to create valid Json Runtime config for trouter client
     * with a single value inside
     * {
     *     "TrouterConnectionUrl": "https://{hostname}/v3/c"
     * }
     *
     * Intended for internal use only thus not public
     */
    static String getJsonConfigWithTrouterUrl(String hostname) {
        return "{\"TrouterConnectionUrl\":\"https://" + hostname + "/v3/c\"}";
    }
}
