// Copyright (c) Microsoft. All rights reserved.

package com.microsoft.trouterclient;

import android.content.Context;

/**
 * This Java class provides entry point to initializing and creating instances of Trouter Client
 * that are standalone and not linked to SlimCore calling stack
 */
public class TrouterClientHost {
    private static TrouterClientHost instance = null;

    private String clientVersion;

    private TrouterClientHost(String clientVersion) {
        this.clientVersion = clientVersion;
    }

    /**
     * Creates new instance of self hosted trouter client
     * It has to be started in order to establish connection to server
     * Two phase startup is designed to let application register listeners prior to connecting
     * - authHeadersProvider is a must have dependency
     * - dataCache is an optional dependency which is recommended for maximum efficiency
     * - trouterHostname is an optional parameter pointing to one of the trouter service clouds (deployments)
     *                   used to separate Consumer vs Enterprise traffic
     *                   sample values are 'go.trouter.skype.com', 'go.trouter.teams.microsoft.com', 'go-us.trouter.teams.microsoft.com'
     * @param authHeadersProvider Authentication provider
     * @param dataCache Connection cache provider
     * @param trouterHostname Trouter service hostname to connect to
     * @return Created client instance
     */
    public ISelfHostedTrouterClient createTrouterClient(ITrouterAuthHeadersProvider authHeadersProvider, ITrouterConnectionDataCache dataCache, String trouterHostname) {
        String runtimeConfig = "";
        if (TrouterUtils.isValidTrouterUrlHostname(trouterHostname)) {
            runtimeConfig = TrouterUtils.getJsonConfigWithTrouterUrl(trouterHostname);
        }

        return (ISelfHostedTrouterClient)createTrouterClientHostPair(clientVersion, authHeadersProvider, dataCache, runtimeConfig);
    }

    /**
     * Initializes native stack of the trouter client
     * Ensures initialization is done once per process
     *
     * - clientVersion must conform to https://dev.azure.com/domoreexp/Teamspace/_wiki/wikis/Teamspace.wiki/7258/Skype-Consumer-Client-IDs-(aka-Platform-IDs)
     *
     * @param context Android app context
     * @param clientVersion Host application version
     * @return singleton instance of TrouterClientHost that can be used to create trouter clients.
     */
    public static TrouterClientHost initialize(Context context, String clientVersion) {
        if (instance == null) {
            try {
                System.loadLibrary("skypert");
            } catch (UnsatisfiedLinkError e) {}
            System.loadLibrary("TrouterClientAndroid");

            nativeInit();

            instance = new TrouterClientHost(clientVersion);
        }
        return instance;
    }

    private static native void nativeInit();

    private native Object createTrouterClientHostPair(
        String clientVersion,
        ITrouterAuthHeadersProvider authHeadersProvider,
        ITrouterConnectionDataCache dataCache,
        String trouterRuntimeConfig);
}

class TrouterAuthHeadersSetter implements ITrouterAuthHeadersSetter {
    private long nativeTrouterPtr;

    public TrouterAuthHeadersSetter(long nativeTrouterPtr) {
        this.nativeTrouterPtr = nativeTrouterPtr;
    }

    @Override
    public void set(String authHeaders) {
        setTrouterAuthHeaders(nativeTrouterPtr, authHeaders);
    }

    private native void setTrouterAuthHeaders(long nativeTrouterPtr, String authHeaders);
}
