// Copyright (c) Microsoft. All rights reserved.

package com.microsoft.trouterclient;

import java.util.List;

/**
 * Interface for a listener that receives and responds to Trouter messages.
 */
public interface ITrouterListener {
    /**
     * Called when Trouter connection is (re-)established
     *
     * This is important mostly because of the `endpointUrl` as the URL where
     * messages are sent to by the originating service(s). Only after receiving
     * this callback and learning the Endpoint URL can the application register
     * the established channel for receiving messages.
     *
     * The Endpoint URL is intended for service-to-service communication, only
     * whitelisted authenticated services can access it and initiate Trouter
     * requests. Use `c2cUrlBase` from `connectionInfo` to obtain a similar URL
     * for client-to-client communication.
     *
     * Note: Called from an internal worker thread, listeners should offload
     *       any longer processing asynchronously.
     *
     * @param endpointUrl    A URL used for routing messages from services to
     *                       this listener
     * @param connectionInfo Additional Trouter connection information
     */
    public void onTrouterConnected(String endpointUrl, ITrouterConnectionInfo connectionInfo);

    /**
     * Signals that Trouter client has become disconnected from the Trouter service.
     * This method is invoked on an internal thread. Please do not block it with any longer activities.
     */
    public void onTrouterDisconnected();

    /**
     * Called when a request is identified for this listener based on the routing path.
     * This method is invoked on an internal thread. Please do not block it with any longer activities.
     * If you don't respond the request in time, the implementation will reply with a default HTTP 504 response after
     * some timeout.
     * @param request A request containing caller's details, as received by the server.
     * @param response Response object to be filled in and sent by the client.
     */
    public void onTrouterRequest(ITrouterRequest request, ITrouterResponse response);

    /**
     * Called when a response has been sent.
     * Can be useful when you are waiting on network I/O completion.
     * This method is invoked on an internal thread. Please do not block it with any longer activities.
     * @param response Trouter response for which this callback was intended; you should not use anything else than its
     *                 getID() method at this time.
     * @param isSuccess true if the response was sent successfully, false otherwise.
     */
    public void onTrouterResponseSent(ITrouterResponse response, boolean isSuccess);

    /**
     * Called when the user activity state change (sent from this client) has been acknowledged by the server.
     * @param correlationVector The correlation vector of the activity state change message that was sent by this client.
     * Note that this is the full outgoing vector, not just the base passed into ITrouter::SetUserActivityState.
     */
    public void onTrouterUserActivityStateAccepted(String correlationVector);

    /**
     * Called when some messages have been dropped on Trouter
     *
     * This can be called at any time by Trouter server. Clients can handle this
     * by manually fetching the latest messages, or in other appropriate way.
     *
     * Note that this will be called only once per a group of flow tags, so it is
     * the client's responsibility to make sure the information doesn't get lost
     * and messages are synced when they should be. This means any retry logic,
     * crash recovery, or possibly storing the current sync state needs to be
     * handled by the client.
     *
     * Some of the examples of the message loss:
     * The message loss might occur when client queues got full and messages are rejected
     * The message to be delivered are already expired.
     *
     * @param flowTags Flow tags representing the type of messages being lost
     */
    public void onTrouterMessageLoss(List<String> flowTags);
}
