// Copyright (c) Microsoft. All rights reserved.

package com.microsoft.trouterclient;

/**
 * This Java interface provides only a subset of the methods available on the underlying C++ object.
 */
public abstract class ITrouter {
    public static ITrouter create(long nativeTrouterPtr) {
        return new Trouter(nativeTrouterPtr);
    }

    /**
     * Registers a callback for receiving Trouter messages.
     * The callback provides a routing path which is used to direct matching server requests to this listener.
     * @param listener ITrouterListener instance to register
     * @param path URL-like path to associate the listener with
     * @param loggingName Identifier for debugging purposes
     * @return true if this callback was registered (even if this path has already been registered), false if
     *         the path contains invalid URL characters.
     */
    public abstract boolean registerListener(ITrouterListener listener, String path, String loggingName);

    /**
     * Unregisters a listener.
     * No further callbacks will occur on the listener after this method returns.
     * @param listener ITrouterListener instance to unregister
     * @return true if unregistered successfully, false if listener was already unregistered or on other error.
     */
    public abstract boolean unregisterListener(ITrouterListener listener);

    /**
     * Sets the user activity state on this endpoint/device
     *
     * The activity state will be sent to the service if it is different
     * from the previously sent state, upon reconnect, or under other
     * circumstances fully in the discretion of the client.
     *
     * The activity will be sent with a newly generated correlation vector on each call.
     *
     * @param state The new user activity state
     */
    public abstract void setUserActivityState(UserActivityState state);

    /**
     * Sets the user activity state on this endpoint/device
     *
     * The activity state will be sent to the service if it is different
     * from the previously sent state, upon reconnect, or under other
     * circumstances fully in the discretion of the client.
     *
     * The activity will be sent with a correlation vector extended from the one
     * passed into this function, and incremented as necessary.
     *
     * @param state The new user activity state
     * @param baseCorrelationVector Base correlation vector to associate with this change
     */
    public abstract void setUserActivityState(UserActivityState state, String baseCorrelationVector);
}
