/*
 * Copyright Microsoft Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microsoft.azure.storage.queue;

import com.microsoft.azure.storage.GeneratedStorageClient;
import com.microsoft.rest.v2.http.HttpPipeline;
import com.microsoft.rest.v2.http.UrlBuilder;
import com.microsoft.rest.v2.policy.DecodingPolicyFactory;
import com.microsoft.rest.v2.policy.RequestPolicyFactory;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;

/**
 * Represents a URL to a Azure storage object. Typically this class is only needed to generate a new pipeline. In most
 * cases, one of the other URL types will be more useful.
 */
public abstract class StorageURL {

    protected final GeneratedStorageClient storageClient;

    protected StorageURL(URL url, HttpPipeline pipeline) {
        if (url == null) {
            throw new IllegalArgumentException("url cannot be null.");
        }
        if (pipeline == null) {
            throw new IllegalArgumentException("pipeline cannot be null.");
        }

        this.storageClient = new GeneratedStorageClient(pipeline)
                .withVersion(Constants.HeaderConstants.TARGET_STORAGE_VERSION);
        this.storageClient.withUrl(url.toString());
    }

    @Override
    public String toString() {
        return this.storageClient.url();
    }

    /**
     * @return The underlying url to the resource.
     */
    public URL toURL() {
        try {
            return new URL(this.storageClient.url());
        } catch (MalformedURLException e) {
            // TODO: remove and update getLeaseId.
        }
        return null;
    }

    /**
     * Appends a string to the end of a URL's path (prefixing the string with a '/' if required).
     *
     * @param baseURL
     *         The url to which the name should be appended.
     * @param name
     *         The name to be appended.
     *
     * @return A url with the name appended.
     *
     * @throws MalformedURLException
     *         Appending the specified name produced an invalid URL.
     */
    protected static URL appendToURLPath(URL baseURL, String name) throws MalformedURLException {
        UrlBuilder url = UrlBuilder.parse(baseURL.toString());
        if (url.path() == null) {
            url.withPath("/"); // .path() will return null if it is empty, so we have to process separately from below.
        } else if (url.path().charAt(url.path().length() - 1) != '/') {
            url.withPath(url.path() + '/');
        }
        url.withPath(url.path() + name);
        return new URL(url.toString());
    }

    /**
     * Creates an pipeline to process the HTTP requests and Responses.
     *
     * @return The pipeline.
     */
    public static HttpPipeline createPipeline() {
        return createPipeline(new AnonymousCredentials(), new PipelineOptions());
    }

    /**
     * Creates an pipeline to process the HTTP requests and Responses.
     *
     * @param pipelineOptions
     *         Configurations for each policy in the pipeline.
     *
     * @return The pipeline.
     */
    public static HttpPipeline createPipeline(PipelineOptions pipelineOptions) {
        return createPipeline(new AnonymousCredentials(), pipelineOptions);
    }

    /**
     * Creates an pipeline to process the HTTP requests and Responses.
     *
     * @param credentials
     *         The credentials the pipeline will use to authenticate the requests.
     *
     * @return The pipeline.
     */
    public static HttpPipeline createPipeline(ICredentials credentials) {
        return createPipeline(credentials, new PipelineOptions());
    }

    /**
     * Creates an pipeline to process the HTTP requests and Responses.
     *
     * @param credentials
     *         The credentials the pipeline will use to authenticate the requests.
     * @param pipelineOptions
     *         Configurations for each policy in the pipeline.
     *
     * @return The pipeline.
     */
    public static HttpPipeline createPipeline(ICredentials credentials, PipelineOptions pipelineOptions) {
        /*
        PipelineOptions is mutable, but its fields refer to immutable objects. This method can pass the fields to other
        methods, but the PipelineOptions object itself can only be used for the duration of this call; it must not be
        passed to anything with a longer lifetime.
         */
        if (credentials == null) {
            throw new IllegalArgumentException(
                    "Credentials cannot be null. For anonymous access use Anonymous Credentials.");
        }
        if (pipelineOptions == null) {
            throw new IllegalArgumentException("pipelineOptions cannot be null. You must at least specify a client.");
        }

        // Closest to API goes first, closest to wire goes last.
        ArrayList<RequestPolicyFactory> factories = new ArrayList<>();
        factories.add(new TelemetryFactory(pipelineOptions.telemetryOptions()));
        factories.add(new RequestIDFactory());
        //factories.add(new RequestRetryFactory(pipelineOptions.requestRetryOptions));
        if (!(credentials instanceof AnonymousCredentials)) {
            factories.add(credentials);
        }
        factories.add(new DecodingPolicyFactory());
        factories.add(new LoggingFactory(pipelineOptions.loggingOptions()));

        return HttpPipeline.build(pipelineOptions.client(),
                factories.toArray(new RequestPolicyFactory[factories.size()]));
    }
}
